/*
 * Copyright 2019-2024 u-blox
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/* Only #includes of u_* and the C standard library are allowed here,
 * no platform stuff and no OS stuff.  Anything required from
 * the platform/OS must be brought in through u_port* to maintain
 * portability.
 */

/** @file
 * @brief Tests that should be run as a postamble in any suite
 * of tests to make sure that everything has been tidied up.
 * This test suite can be made to run last by setting
 * U_RUNNER_POSTAMBLE_STR to "postamble", which runner.c sets it to
 * by default anyway.
 */

#ifdef U_CFG_OVERRIDE
# include "u_cfg_override.h" // For a customer's configuration override
#endif

#include "stddef.h"    // NULL, size_t etc.
#include "stdint.h"    // int32_t etc.
#include "stdbool.h"

#include "u_cfg_sw.h"
#include "u_cfg_os_platform_specific.h"
#include "u_cfg_app_platform_specific.h"
#include "u_cfg_test_platform_specific.h"

#include "u_error_common.h"

#include "u_port.h"
#include "u_port_debug.h"
#include "u_port_os.h"

#include "u_test_util_resource_check.h"

/* ----------------------------------------------------------------
 * COMPILE-TIME MACROS
 * -------------------------------------------------------------- */

/** The string to put at the start of all prints from this test.
 */
#define U_TEST_PREFIX "U_POSTAMBLE_TEST: "

/** Print a whole line, with terminator, prefixed for this test file.
 */
#define U_TEST_PRINT_LINE(format, ...) uPortLog(U_TEST_PREFIX format "\n", ##__VA_ARGS__)

/* ----------------------------------------------------------------
 * TYPES
 * -------------------------------------------------------------- */

/* ----------------------------------------------------------------
 * VARIABLES
 * -------------------------------------------------------------- */

/* ----------------------------------------------------------------
 * STATIC FUNCTIONS
 * -------------------------------------------------------------- */

/* ----------------------------------------------------------------
 * PUBLIC FUNCTIONS: TESTS
 * -------------------------------------------------------------- */

/** The postamble: check resources.
 */
U_PORT_TEST_FUNCTION("[postamble]", "postambleResourceCheck")
{
    if (!uTestUtilResourceCheck(U_TEST_PREFIX,
                                U_TEST_UTIL_RESOURCE_CHECK_ERROR_MARKER,
                                true)) {
        U_TEST_PRINT_LINE("too many resources outstanding.");
        if (uTestUtilGetNumFailed() > 0) {
            U_TEST_PRINT_LINE("this _might_ be because of test failures"
                              " resulting in clean-ups being skipped but"
                              " you MUST check.");
        } else {
            // Assert only if the number of failed tests is zero; this is
            // because clean-up is inevitably skpped when a test failure
            // occurs and so bleating about resources only adds more
            // needless noise: of _course_ there will be leaked resources!
            U_PORT_TEST_ASSERT(false);
        }
    } else {
        U_TEST_PRINT_LINE("no resources outstanding.");
    }
}

// End of file
