#! /usr/bin/python3

"""
This script takes output from objdump tool of an ELF file, and generates a
compilable C file. The C file describes a struct comprised of the functions in
the ELF file, serving as a library descriptor. 

Input is read from a file. This file should be generated something like this:

$ objdump -tj .text mylibrary.elf > mylibrary.sym

"""

import sys
import os
import re

def emit(name, version, flags, length, syms):
  """ Emit source code for symbols """
  print("/* autogenerated C file */")
  emitBegin(name, version, flags, length, syms)
  for symName in syms:
    emitEntry(symName, syms[symName])
  emitEnd(name, version, flags, length, syms)

def emitBegin(name, version, flags, length, syms):
  """ Emit source code for symbols: beginning """
  print("#include \"u_lib_internal.h\"")
  print("")
  print("const __attribute__(( section (\".ulibhdr\") )) _uLibHeader_t uLibHdr{} = {}".format(name, "{"))
  print("    .magic = 0xc01df00d,")
  print('    .name = "{}",'.format(name))
  print("    .version = {},".format(version))
  print("    .flags = {} | (U_LIB_ARCH << U_LIB_HDR_FLAG_ARCH_BITPOS),".format(flags))
  print("    .count = {},".format(len(syms)))
  print("    .length = {},".format(length))
  print("};")
  print("const  __attribute__(( section (\".ulibtbl\") )) _uLibFunctionDescriptor_t uLibSymFuncList{}[] = {}".format(name, "{"))

def emitEntry(symName, symProps):
  """ Emit source code for symbols: per symbol entry """
  flags = 0
  if symProps["type"] == "F":
    flags = flags | (1<<0) # function
  if symName == "___libOpen":
    flags = flags | (1<<1) # library initialiser
  if symName == "___libClose":
    flags = flags | (1<<2) # library finaliser
  print('    {} .name = "{}", .offset = {}, .flags = {} {},'.format("{", symName, symProps["offset"], flags, "}"))

def emitEnd(name, version, flags, length, syms):
  """ Emit source code for symbols: end """
  print("};")

def main():
  """
  Build a dict from passed input file, generate code from it
  and spit out a C file.
  """
  if len(sys.argv) <= 1:
    print("Need path to symbol file.")
    print("Generate it something like this:")
    print("$ objdump -tj .text mylibrary.elf > mylibrary.sym")
    print("Exiting...")
    sys.exit()

  filepath = sys.argv[1]

  if not os.path.isfile(filepath):
    print("File {} does not exist. Exiting...".format(filepath))
    sys.exit()
  
  # fetch global symbols from .text section as given by objdump, ex:
  #   SYMBOL TABLE:
  #   00000154 l    d  .text	00000000 .text
  #   000001b8 g     F .text	0000000a libClose
  #   00000194 g     F .text	00000024 libOpen
  #   00000184 g     F .text	00000010 libFibLastRes
  #   00000154 g     F .text	0000002e libFibCalc
  #
  # also eats key/value pairs, like 
  #   KEY1 = VALUE2
  # usable for passing extra information on the library
  syms = dict()
  extras = dict()
  with open(filepath) as fp:
    textOffset = 0
    pentry = re.compile(r'([0-9A-Fa-f]{8,16})\s([g|l]).....(.)\s\.text\s*([0-9A-Fa-f]{8,16})\s(.*)')
    pextra = re.compile(r'(\w+)\s*=\s*(\w+)')

    for line in fp:
      # is this an objdump entry?
      m = pentry.match(line)
      if m != None:
        if m.group(2) == "l" and m.group(3) == " " and m.group(5) == ".text":
          # fetch the first .text offset, indicating start of functions
          textOffset = int(m.group(1),16)
        elif m.group(2) == "g":
          # found a symbol, put it in dict at given address - .text offset
          syms[m.group(5)] = {"offset":int(m.group(1),16)-textOffset,"type":m.group(3),"len":int(m.group(4),16)}
        continue
      # or is this mayhap some extra info?
      m = pextra.match(line)
      if m != None:
        extras[m.group(1)] = m.group(2)

  # emit C code generated symbol table
  emit(extras["name"], int(extras["version"]), int(extras["flags"]), int(extras["length"]), syms)

if __name__ == '__main__':
   main()
