/*
 * Copyright 2019-2024 u-blox
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/* Only #includes of u_* and the C standard library are allowed here,
 * no platform stuff and no OS stuff.  Anything required from
 * the platform/OS must be brought in through u_port* to maintain
 * portability.
 */

/** @file
 * @brief Test data for the u-blox security credential API testing.
 */

#ifdef U_CFG_OVERRIDE
# include "u_cfg_override.h" // For a customer's configuration override
#endif

#include "stddef.h"    // NULL, size_t etc.
#include "stdint.h"    // int32_t etc.
#include "stdbool.h"

#ifdef U_SECURITY_CREDENTIAL_TEST_FORMATS
#include "u_security_credential.h"
#endif

#include "u_security_credential_test_data.h"

/* ----------------------------------------------------------------
 * COMPILE-TIME MACROS
 * -------------------------------------------------------------- */

/* ----------------------------------------------------------------
 * TYPES
 * -------------------------------------------------------------- */

/* ----------------------------------------------------------------
 * VARIABLES: 1024-BIT KEY
 * -------------------------------------------------------------- */

// A test PKCS1-encoded encrypted PEM-format private key generated
// using OpenSSL as follows:
//
// openssl genrsa -des3 -out key_1024_pkcs1.pem 1024
// ...with passphrase "ubxlib".
//
// See key_1024_pkcs1.pem.
const uint8_t gUSecurityCredentialTestKey1024Pkcs1Pem[] = {0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x42, 0x45, 0x47,
                                                           0x49, 0x4e, 0x20, 0x52, 0x53, 0x41, 0x20, 0x50,
                                                           0x52, 0x49, 0x56, 0x41, 0x54, 0x45, 0x20, 0x4b,
                                                           0x45, 0x59, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x0d,
                                                           0x0a, 0x50, 0x72, 0x6f, 0x63, 0x2d, 0x54, 0x79,
                                                           0x70, 0x65, 0x3a, 0x20, 0x34, 0x2c, 0x45, 0x4e,
                                                           0x43, 0x52, 0x59, 0x50, 0x54, 0x45, 0x44, 0x0d,
                                                           0x0a, 0x44, 0x45, 0x4b, 0x2d, 0x49, 0x6e, 0x66,
                                                           0x6f, 0x3a, 0x20, 0x44, 0x45, 0x53, 0x2d, 0x45,
                                                           0x44, 0x45, 0x33, 0x2d, 0x43, 0x42, 0x43, 0x2c,
                                                           0x36, 0x45, 0x39, 0x39, 0x45, 0x35, 0x36, 0x38,
                                                           0x36, 0x34, 0x42, 0x36, 0x32, 0x44, 0x45, 0x45,
                                                           0x0d, 0x0a, 0x0d, 0x0a, 0x67, 0x56, 0x51, 0x4c,
                                                           0x55, 0x37, 0x34, 0x32, 0x72, 0x57, 0x69, 0x6e,
                                                           0x70, 0x56, 0x68, 0x33, 0x64, 0x5a, 0x4e, 0x68,
                                                           0x6c, 0x6f, 0x34, 0x43, 0x61, 0x5a, 0x77, 0x4f,
                                                           0x6b, 0x4b, 0x44, 0x34, 0x71, 0x6e, 0x68, 0x68,
                                                           0x42, 0x49, 0x6f, 0x4b, 0x36, 0x68, 0x59, 0x6c,
                                                           0x44, 0x4c, 0x43, 0x31, 0x63, 0x7a, 0x33, 0x62,
                                                           0x71, 0x7a, 0x73, 0x43, 0x48, 0x34, 0x73, 0x68,
                                                           0x6b, 0x70, 0x32, 0x77, 0x0d, 0x0a, 0x63, 0x5a,
                                                           0x63, 0x44, 0x6a, 0x4f, 0x32, 0x65, 0x69, 0x2f,
                                                           0x77, 0x69, 0x41, 0x6a, 0x78, 0x41, 0x62, 0x79,
                                                           0x42, 0x78, 0x78, 0x66, 0x64, 0x64, 0x4c, 0x50,
                                                           0x4b, 0x50, 0x35, 0x70, 0x74, 0x72, 0x76, 0x38,
                                                           0x65, 0x46, 0x57, 0x79, 0x34, 0x66, 0x2f, 0x75,
                                                           0x39, 0x49, 0x6a, 0x63, 0x44, 0x74, 0x5a, 0x77,
                                                           0x71, 0x48, 0x4f, 0x35, 0x2f, 0x56, 0x4b, 0x35,
                                                           0x4d, 0x6a, 0x36, 0x75, 0x79, 0x48, 0x0d, 0x0a,
                                                           0x4a, 0x38, 0x58, 0x4b, 0x2f, 0x39, 0x4e, 0x6b,
                                                           0x31, 0x31, 0x69, 0x73, 0x68, 0x35, 0x59, 0x37,
                                                           0x59, 0x57, 0x34, 0x72, 0x54, 0x33, 0x47, 0x36,
                                                           0x6c, 0x6f, 0x70, 0x50, 0x55, 0x54, 0x2b, 0x57,
                                                           0x55, 0x66, 0x78, 0x35, 0x41, 0x76, 0x77, 0x6b,
                                                           0x53, 0x55, 0x46, 0x6d, 0x46, 0x69, 0x6f, 0x38,
                                                           0x33, 0x41, 0x49, 0x78, 0x49, 0x62, 0x67, 0x70,
                                                           0x75, 0x74, 0x4a, 0x36, 0x64, 0x71, 0x51, 0x35,
                                                           0x0d, 0x0a, 0x6e, 0x6f, 0x5a, 0x39, 0x57, 0x4e,
                                                           0x4a, 0x4b, 0x74, 0x4f, 0x67, 0x59, 0x46, 0x57,
                                                           0x73, 0x43, 0x46, 0x35, 0x73, 0x4b, 0x68, 0x6a,
                                                           0x42, 0x2f, 0x73, 0x48, 0x53, 0x64, 0x41, 0x35,
                                                           0x58, 0x54, 0x79, 0x55, 0x5a, 0x69, 0x66, 0x63,
                                                           0x43, 0x4e, 0x65, 0x4d, 0x39, 0x48, 0x62, 0x55,
                                                           0x39, 0x73, 0x73, 0x4f, 0x69, 0x6d, 0x34, 0x74,
                                                           0x55, 0x4e, 0x73, 0x42, 0x70, 0x37, 0x47, 0x63,
                                                           0x74, 0x33, 0x0d, 0x0a, 0x6a, 0x56, 0x79, 0x55,
                                                           0x4d, 0x48, 0x48, 0x35, 0x4c, 0x75, 0x74, 0x53,
                                                           0x68, 0x57, 0x2b, 0x69, 0x35, 0x6f, 0x72, 0x42,
                                                           0x62, 0x5a, 0x39, 0x66, 0x7a, 0x51, 0x69, 0x65,
                                                           0x79, 0x43, 0x4e, 0x57, 0x42, 0x4f, 0x38, 0x59,
                                                           0x50, 0x39, 0x6e, 0x55, 0x4d, 0x72, 0x62, 0x46,
                                                           0x7a, 0x6c, 0x72, 0x62, 0x73, 0x67, 0x64, 0x32,
                                                           0x4e, 0x53, 0x4f, 0x53, 0x4c, 0x4e, 0x4c, 0x59,
                                                           0x2b, 0x59, 0x49, 0x35, 0x0d, 0x0a, 0x52, 0x4d,
                                                           0x56, 0x72, 0x63, 0x34, 0x4f, 0x61, 0x6a, 0x74,
                                                           0x36, 0x41, 0x74, 0x7a, 0x54, 0x39, 0x54, 0x57,
                                                           0x6a, 0x70, 0x47, 0x36, 0x6e, 0x6c, 0x37, 0x62,
                                                           0x44, 0x5a, 0x42, 0x36, 0x51, 0x43, 0x5a, 0x62,
                                                           0x6e, 0x72, 0x74, 0x73, 0x43, 0x56, 0x41, 0x32,
                                                           0x35, 0x49, 0x44, 0x70, 0x72, 0x35, 0x42, 0x34,
                                                           0x54, 0x56, 0x64, 0x6c, 0x63, 0x6c, 0x44, 0x31,
                                                           0x61, 0x37, 0x44, 0x61, 0x5a, 0x43, 0x0d, 0x0a,
                                                           0x62, 0x43, 0x6b, 0x70, 0x4f, 0x4c, 0x61, 0x74,
                                                           0x55, 0x53, 0x68, 0x53, 0x6a, 0x42, 0x75, 0x32,
                                                           0x59, 0x69, 0x49, 0x63, 0x6d, 0x5a, 0x4e, 0x6d,
                                                           0x6b, 0x38, 0x43, 0x4b, 0x62, 0x38, 0x45, 0x6a,
                                                           0x4e, 0x6d, 0x30, 0x30, 0x62, 0x51, 0x53, 0x7a,
                                                           0x2f, 0x6a, 0x37, 0x48, 0x64, 0x63, 0x52, 0x57,
                                                           0x38, 0x4b, 0x48, 0x41, 0x45, 0x31, 0x7a, 0x56,
                                                           0x76, 0x35, 0x70, 0x57, 0x46, 0x51, 0x53, 0x43,
                                                           0x0d, 0x0a, 0x65, 0x6f, 0x42, 0x45, 0x74, 0x51,
                                                           0x74, 0x2f, 0x6d, 0x44, 0x47, 0x55, 0x39, 0x6d,
                                                           0x53, 0x59, 0x5a, 0x78, 0x45, 0x38, 0x6f, 0x71,
                                                           0x55, 0x62, 0x4d, 0x31, 0x6a, 0x74, 0x51, 0x56,
                                                           0x31, 0x73, 0x6e, 0x67, 0x59, 0x4e, 0x53, 0x44,
                                                           0x46, 0x47, 0x74, 0x2b, 0x58, 0x50, 0x57, 0x7a,
                                                           0x6c, 0x50, 0x71, 0x45, 0x34, 0x66, 0x4c, 0x77,
                                                           0x48, 0x73, 0x34, 0x30, 0x4c, 0x6f, 0x57, 0x78,
                                                           0x6e, 0x46, 0x0d, 0x0a, 0x6e, 0x47, 0x64, 0x61,
                                                           0x75, 0x6c, 0x77, 0x46, 0x42, 0x49, 0x52, 0x64,
                                                           0x58, 0x2f, 0x6c, 0x6e, 0x61, 0x57, 0x55, 0x6e,
                                                           0x57, 0x43, 0x41, 0x67, 0x56, 0x70, 0x73, 0x6b,
                                                           0x72, 0x7a, 0x61, 0x59, 0x66, 0x79, 0x6a, 0x58,
                                                           0x42, 0x6a, 0x54, 0x46, 0x4b, 0x31, 0x65, 0x45,
                                                           0x78, 0x2b, 0x79, 0x65, 0x68, 0x4d, 0x78, 0x4b,
                                                           0x68, 0x75, 0x4b, 0x72, 0x69, 0x54, 0x6d, 0x52,
                                                           0x31, 0x54, 0x58, 0x45, 0x0d, 0x0a, 0x51, 0x57,
                                                           0x6d, 0x71, 0x51, 0x4c, 0x4c, 0x77, 0x51, 0x51,
                                                           0x68, 0x77, 0x37, 0x72, 0x2f, 0x47, 0x50, 0x67,
                                                           0x31, 0x76, 0x58, 0x33, 0x49, 0x54, 0x78, 0x30,
                                                           0x6a, 0x38, 0x42, 0x64, 0x69, 0x67, 0x71, 0x32,
                                                           0x74, 0x51, 0x61, 0x75, 0x73, 0x34, 0x66, 0x32,
                                                           0x63, 0x38, 0x2b, 0x57, 0x32, 0x4e, 0x45, 0x63,
                                                           0x6a, 0x4b, 0x4c, 0x30, 0x39, 0x41, 0x2b, 0x7a,
                                                           0x2f, 0x6a, 0x66, 0x42, 0x4a, 0x6e, 0x0d, 0x0a,
                                                           0x79, 0x59, 0x79, 0x6e, 0x52, 0x6d, 0x47, 0x33,
                                                           0x4a, 0x52, 0x2b, 0x53, 0x4f, 0x70, 0x57, 0x6a,
                                                           0x52, 0x48, 0x2f, 0x41, 0x30, 0x79, 0x51, 0x45,
                                                           0x4d, 0x46, 0x59, 0x78, 0x58, 0x2f, 0x4c, 0x4c,
                                                           0x72, 0x71, 0x71, 0x67, 0x41, 0x41, 0x77, 0x45,
                                                           0x73, 0x74, 0x2f, 0x53, 0x68, 0x51, 0x64, 0x5a,
                                                           0x2f, 0x56, 0x2f, 0x2b, 0x32, 0x6e, 0x4b, 0x70,
                                                           0x61, 0x6e, 0x67, 0x2f, 0x54, 0x79, 0x53, 0x56,
                                                           0x0d, 0x0a, 0x30, 0x32, 0x77, 0x74, 0x70, 0x72,
                                                           0x65, 0x7a, 0x77, 0x74, 0x53, 0x49, 0x62, 0x4a,
                                                           0x51, 0x71, 0x2f, 0x57, 0x75, 0x4e, 0x39, 0x6e,
                                                           0x65, 0x33, 0x52, 0x59, 0x75, 0x6b, 0x49, 0x65,
                                                           0x4f, 0x41, 0x53, 0x50, 0x33, 0x6c, 0x46, 0x45,
                                                           0x36, 0x53, 0x44, 0x54, 0x35, 0x6b, 0x43, 0x74,
                                                           0x61, 0x35, 0x56, 0x51, 0x79, 0x4f, 0x4a, 0x6a,
                                                           0x33, 0x6d, 0x61, 0x34, 0x70, 0x65, 0x4d, 0x30,
                                                           0x32, 0x43, 0x0d, 0x0a, 0x6f, 0x46, 0x78, 0x45,
                                                           0x79, 0x37, 0x31, 0x37, 0x6f, 0x6a, 0x77, 0x61,
                                                           0x35, 0x39, 0x33, 0x45, 0x63, 0x33, 0x51, 0x56,
                                                           0x34, 0x76, 0x33, 0x34, 0x58, 0x71, 0x69, 0x52,
                                                           0x72, 0x6d, 0x74, 0x79, 0x41, 0x56, 0x5a, 0x39,
                                                           0x72, 0x4b, 0x7a, 0x68, 0x39, 0x76, 0x4d, 0x3d,
                                                           0x0d, 0x0a, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x45,
                                                           0x4e, 0x44, 0x20, 0x52, 0x53, 0x41, 0x20, 0x50,
                                                           0x52, 0x49, 0x56, 0x41, 0x54, 0x45, 0x20, 0x4b,
                                                           0x45, 0x59, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x0d,
                                                           0x0a
                                                          };

// Number of items in the gUSecurityCredentialTestKey1024Pkcs1Pem array.
const size_t gUSecurityCredentialTestKey1024Pkcs1PemSize =
    sizeof(gUSecurityCredentialTestKey1024Pkcs1Pem) /
    sizeof(gUSecurityCredentialTestKey1024Pkcs1Pem[0]);

// A decrypted version of gUSecurityCredentialTestKey1024Pkcs1Pem
// generated using OpenSSL as follows:
//
// openssl pkey -in key_1024_pkcs1.pem -out key_1024_pkcs1_no_pass.pem
// ...with passphrase "ubxlib".
//
//lint -esym(552, gUSecurityCredentialTestKey1024Pkcs1PemNoPass)
// Suppress variable not referenced.
const uint8_t gUSecurityCredentialTestKey1024Pkcs1PemNoPass[] = {0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x42, 0x45, 0x47,
                                                                 0x49, 0x4e, 0x20, 0x52, 0x53, 0x41, 0x20, 0x50,
                                                                 0x52, 0x49, 0x56, 0x41, 0x54, 0x45, 0x20, 0x4b,
                                                                 0x45, 0x59, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x0a,
                                                                 0x4d, 0x49, 0x49, 0x43, 0x57, 0x77, 0x49, 0x42,
                                                                 0x41, 0x41, 0x4b, 0x42, 0x67, 0x51, 0x43, 0x77,
                                                                 0x6e, 0x78, 0x55, 0x4f, 0x6a, 0x72, 0x4a, 0x74,
                                                                 0x46, 0x34, 0x37, 0x4c, 0x41, 0x36, 0x59, 0x66,
                                                                 0x76, 0x54, 0x64, 0x45, 0x43, 0x73, 0x37, 0x38,
                                                                 0x45, 0x47, 0x52, 0x46, 0x59, 0x30, 0x34, 0x6e,
                                                                 0x5a, 0x49, 0x64, 0x66, 0x68, 0x36, 0x51, 0x55,
                                                                 0x68, 0x42, 0x47, 0x39, 0x32, 0x75, 0x69, 0x73,
                                                                 0x0a, 0x33, 0x79, 0x78, 0x79, 0x31, 0x31, 0x6a,
                                                                 0x38, 0x78, 0x52, 0x56, 0x56, 0x63, 0x4a, 0x59,
                                                                 0x37, 0x6b, 0x4c, 0x41, 0x4d, 0x4b, 0x79, 0x77,
                                                                 0x57, 0x4c, 0x30, 0x55, 0x32, 0x49, 0x74, 0x62,
                                                                 0x34, 0x78, 0x41, 0x77, 0x39, 0x68, 0x4e, 0x37,
                                                                 0x6c, 0x7a, 0x38, 0x68, 0x6e, 0x4c, 0x4c, 0x4f,
                                                                 0x67, 0x4a, 0x49, 0x66, 0x5a, 0x6b, 0x30, 0x65,
                                                                 0x6c, 0x42, 0x6b, 0x75, 0x68, 0x38, 0x4e, 0x68,
                                                                 0x78, 0x0a, 0x5a, 0x52, 0x39, 0x7a, 0x31, 0x67,
                                                                 0x2f, 0x65, 0x43, 0x30, 0x59, 0x34, 0x6b, 0x51,
                                                                 0x38, 0x44, 0x5a, 0x2b, 0x67, 0x75, 0x49, 0x61,
                                                                 0x61, 0x32, 0x6f, 0x39, 0x4b, 0x57, 0x55, 0x57,
                                                                 0x79, 0x41, 0x52, 0x4f, 0x37, 0x6c, 0x68, 0x58,
                                                                 0x7a, 0x51, 0x66, 0x61, 0x76, 0x42, 0x59, 0x30,
                                                                 0x66, 0x4c, 0x2b, 0x61, 0x61, 0x4d, 0x79, 0x7a,
                                                                 0x34, 0x6d, 0x56, 0x51, 0x49, 0x44, 0x41, 0x51,
                                                                 0x41, 0x42, 0x0a, 0x41, 0x6f, 0x47, 0x41, 0x41,
                                                                 0x38, 0x42, 0x64, 0x51, 0x69, 0x48, 0x6c, 0x79,
                                                                 0x45, 0x74, 0x6e, 0x7a, 0x51, 0x67, 0x57, 0x7a,
                                                                 0x7a, 0x58, 0x46, 0x70, 0x72, 0x78, 0x55, 0x43,
                                                                 0x53, 0x61, 0x50, 0x2b, 0x71, 0x6c, 0x52, 0x54,
                                                                 0x36, 0x54, 0x69, 0x2b, 0x7a, 0x52, 0x32, 0x56,
                                                                 0x37, 0x6b, 0x32, 0x5a, 0x30, 0x48, 0x76, 0x44,
                                                                 0x7a, 0x7a, 0x75, 0x52, 0x4d, 0x49, 0x75, 0x4d,
                                                                 0x62, 0x41, 0x39, 0x0a, 0x38, 0x38, 0x4b, 0x67,
                                                                 0x62, 0x6b, 0x4e, 0x30, 0x65, 0x63, 0x32, 0x77,
                                                                 0x38, 0x67, 0x65, 0x76, 0x41, 0x4b, 0x64, 0x63,
                                                                 0x79, 0x59, 0x76, 0x67, 0x64, 0x71, 0x54, 0x38,
                                                                 0x47, 0x43, 0x6b, 0x55, 0x62, 0x79, 0x77, 0x79,
                                                                 0x4f, 0x34, 0x39, 0x51, 0x71, 0x6f, 0x2b, 0x62,
                                                                 0x55, 0x4f, 0x30, 0x2f, 0x59, 0x30, 0x54, 0x30,
                                                                 0x38, 0x6e, 0x77, 0x33, 0x33, 0x49, 0x72, 0x44,
                                                                 0x30, 0x44, 0x47, 0x71, 0x0a, 0x5a, 0x48, 0x2b,
                                                                 0x49, 0x73, 0x41, 0x4c, 0x39, 0x34, 0x52, 0x67,
                                                                 0x4d, 0x61, 0x78, 0x48, 0x6e, 0x51, 0x6f, 0x48,
                                                                 0x5a, 0x41, 0x47, 0x2f, 0x44, 0x75, 0x31, 0x36,
                                                                 0x6b, 0x2b, 0x6c, 0x41, 0x33, 0x79, 0x63, 0x71,
                                                                 0x34, 0x51, 0x52, 0x48, 0x73, 0x4e, 0x37, 0x35,
                                                                 0x52, 0x32, 0x55, 0x45, 0x43, 0x51, 0x51, 0x44,
                                                                 0x6d, 0x4f, 0x51, 0x77, 0x34, 0x58, 0x51, 0x4b,
                                                                 0x44, 0x41, 0x54, 0x77, 0x48, 0x0a, 0x6b, 0x50,
                                                                 0x6d, 0x4f, 0x68, 0x33, 0x67, 0x6d, 0x42, 0x65,
                                                                 0x39, 0x56, 0x35, 0x49, 0x74, 0x4c, 0x6f, 0x71,
                                                                 0x2f, 0x59, 0x7a, 0x65, 0x43, 0x6f, 0x41, 0x4f,
                                                                 0x5a, 0x68, 0x32, 0x64, 0x4e, 0x47, 0x35, 0x56,
                                                                 0x52, 0x45, 0x54, 0x6a, 0x61, 0x61, 0x65, 0x45,
                                                                 0x50, 0x68, 0x39, 0x66, 0x6f, 0x44, 0x44, 0x33,
                                                                 0x50, 0x6f, 0x41, 0x72, 0x2f, 0x31, 0x45, 0x57,
                                                                 0x74, 0x6a, 0x67, 0x65, 0x55, 0x62, 0x0a, 0x32,
                                                                 0x51, 0x6b, 0x52, 0x76, 0x70, 0x45, 0x48, 0x41,
                                                                 0x6b, 0x45, 0x41, 0x78, 0x47, 0x57, 0x6a, 0x72,
                                                                 0x4b, 0x53, 0x35, 0x34, 0x56, 0x69, 0x77, 0x56,
                                                                 0x5a, 0x45, 0x44, 0x52, 0x31, 0x6b, 0x57, 0x74,
                                                                 0x42, 0x4f, 0x55, 0x59, 0x59, 0x69, 0x5a, 0x59,
                                                                 0x53, 0x58, 0x53, 0x4f, 0x41, 0x73, 0x43, 0x51,
                                                                 0x2b, 0x59, 0x47, 0x79, 0x36, 0x33, 0x71, 0x54,
                                                                 0x59, 0x6c, 0x30, 0x75, 0x6e, 0x34, 0x74, 0x0a,
                                                                 0x39, 0x46, 0x4f, 0x67, 0x4e, 0x58, 0x41, 0x34,
                                                                 0x49, 0x36, 0x35, 0x51, 0x7a, 0x32, 0x2b, 0x57,
                                                                 0x37, 0x7a, 0x52, 0x79, 0x39, 0x70, 0x47, 0x31,
                                                                 0x36, 0x68, 0x4a, 0x62, 0x69, 0x4b, 0x64, 0x69,
                                                                 0x77, 0x77, 0x4a, 0x41, 0x52, 0x4f, 0x74, 0x42,
                                                                 0x43, 0x45, 0x4a, 0x33, 0x74, 0x51, 0x59, 0x53,
                                                                 0x6c, 0x57, 0x56, 0x46, 0x2f, 0x2b, 0x2f, 0x33,
                                                                 0x2f, 0x4e, 0x48, 0x38, 0x41, 0x75, 0x65, 0x47,
                                                                 0x0a, 0x39, 0x35, 0x35, 0x39, 0x4b, 0x39, 0x62,
                                                                 0x39, 0x38, 0x4f, 0x31, 0x4d, 0x43, 0x6e, 0x44,
                                                                 0x74, 0x2b, 0x61, 0x51, 0x2f, 0x5a, 0x4b, 0x39,
                                                                 0x52, 0x55, 0x30, 0x36, 0x37, 0x61, 0x35, 0x4a,
                                                                 0x36, 0x52, 0x4a, 0x35, 0x45, 0x6f, 0x57, 0x6a,
                                                                 0x70, 0x56, 0x42, 0x44, 0x68, 0x36, 0x41, 0x6b,
                                                                 0x44, 0x73, 0x43, 0x46, 0x35, 0x77, 0x4a, 0x45,
                                                                 0x65, 0x30, 0x77, 0x4a, 0x41, 0x4a, 0x38, 0x42,
                                                                 0x75, 0x0a, 0x31, 0x5a, 0x58, 0x74, 0x37, 0x62,
                                                                 0x65, 0x32, 0x30, 0x55, 0x72, 0x74, 0x76, 0x2f,
                                                                 0x59, 0x2f, 0x53, 0x79, 0x6d, 0x66, 0x59, 0x66,
                                                                 0x4a, 0x31, 0x49, 0x52, 0x32, 0x33, 0x70, 0x42,
                                                                 0x4a, 0x4f, 0x58, 0x68, 0x48, 0x6b, 0x46, 0x44,
                                                                 0x76, 0x61, 0x62, 0x65, 0x46, 0x42, 0x6d, 0x44,
                                                                 0x48, 0x4c, 0x2b, 0x57, 0x55, 0x6e, 0x79, 0x30,
                                                                 0x6b, 0x77, 0x67, 0x38, 0x42, 0x58, 0x71, 0x50,
                                                                 0x54, 0x73, 0x0a, 0x4e, 0x76, 0x53, 0x46, 0x78,
                                                                 0x48, 0x53, 0x50, 0x76, 0x70, 0x59, 0x43, 0x6f,
                                                                 0x76, 0x46, 0x47, 0x59, 0x77, 0x4a, 0x41, 0x4b,
                                                                 0x65, 0x33, 0x6a, 0x4d, 0x72, 0x79, 0x74, 0x58,
                                                                 0x64, 0x53, 0x4f, 0x4b, 0x7a, 0x70, 0x57, 0x4f,
                                                                 0x58, 0x4e, 0x69, 0x4e, 0x4b, 0x4b, 0x4a, 0x2f,
                                                                 0x4e, 0x4a, 0x63, 0x67, 0x72, 0x55, 0x7a, 0x31,
                                                                 0x63, 0x39, 0x76, 0x6d, 0x4e, 0x43, 0x6c, 0x32,
                                                                 0x67, 0x33, 0x45, 0x0a, 0x48, 0x6d, 0x33, 0x45,
                                                                 0x4c, 0x30, 0x39, 0x70, 0x41, 0x6a, 0x63, 0x7a,
                                                                 0x30, 0x63, 0x54, 0x4e, 0x4e, 0x55, 0x6b, 0x63,
                                                                 0x6f, 0x5a, 0x44, 0x36, 0x63, 0x4f, 0x31, 0x5a,
                                                                 0x70, 0x31, 0x68, 0x6e, 0x63, 0x66, 0x52, 0x4b,
                                                                 0x4d, 0x70, 0x5a, 0x73, 0x49, 0x41, 0x3d, 0x3d,
                                                                 0x0a, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x45, 0x4e,
                                                                 0x44, 0x20, 0x52, 0x53, 0x41, 0x20, 0x50, 0x52,
                                                                 0x49, 0x56, 0x41, 0x54, 0x45, 0x20, 0x4b, 0x45,
                                                                 0x59, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x0a
                                                                };

// Number of items in the gUSecurityCredentialTestKey1024Pkcs1PemNoPass array.
//lint -esym(552, gUSecurityCredentialTestKey1024Pkcs1PemNoPassSize) Suppress
// variable not referenced.
const size_t gUSecurityCredentialTestKey1024Pkcs1PemNoPassSize =
    sizeof(gUSecurityCredentialTestKey1024Pkcs1PemNoPass) /
    sizeof(gUSecurityCredentialTestKey1024Pkcs1PemNoPass[0]);

// A PKCS8-encoded version of gSecurityCredentialTestKey1024Pkcs1Pem
// (i.e. encrypted) generated using OpenSSL as follows:
//
// openssl pkcs8 -topk8 -in key_1024_pkcs1.pem -out key_1024_pkcs8.pem
// ...with "ubxlib" for both passphrases
//
//lint -esym(552, gUSecurityCredentialTestKey1024Pkcs8Pem) Suppress
// variable not referenced.
const uint8_t gUSecurityCredentialTestKey1024Pkcs8Pem[] = {0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x42, 0x45, 0x47,
                                                           0x49, 0x4e, 0x20, 0x45, 0x4e, 0x43, 0x52, 0x59,
                                                           0x50, 0x54, 0x45, 0x44, 0x20, 0x50, 0x52, 0x49,
                                                           0x56, 0x41, 0x54, 0x45, 0x20, 0x4b, 0x45, 0x59,
                                                           0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x0a, 0x4d, 0x49,
                                                           0x49, 0x43, 0x6f, 0x54, 0x41, 0x62, 0x42, 0x67,
                                                           0x6b, 0x71, 0x68, 0x6b, 0x69, 0x47, 0x39, 0x77,
                                                           0x30, 0x42, 0x42, 0x51, 0x4d, 0x77, 0x44, 0x67,
                                                           0x51, 0x49, 0x61, 0x45, 0x6b, 0x5a, 0x72, 0x47,
                                                           0x45, 0x65, 0x44, 0x75, 0x30, 0x43, 0x41, 0x67,
                                                           0x67, 0x41, 0x42, 0x49, 0x49, 0x43, 0x67, 0x4c,
                                                           0x6d, 0x4c, 0x6b, 0x68, 0x57, 0x79, 0x69, 0x46,
                                                           0x2b, 0x6c, 0x72, 0x77, 0x65, 0x52, 0x0a, 0x32,
                                                           0x58, 0x33, 0x41, 0x71, 0x75, 0x56, 0x38, 0x7a,
                                                           0x50, 0x2b, 0x6e, 0x71, 0x6e, 0x71, 0x2f, 0x6e,
                                                           0x79, 0x68, 0x55, 0x33, 0x39, 0x6e, 0x62, 0x63,
                                                           0x5a, 0x69, 0x33, 0x6a, 0x50, 0x50, 0x4d, 0x70,
                                                           0x61, 0x38, 0x6e, 0x74, 0x35, 0x57, 0x68, 0x50,
                                                           0x6d, 0x34, 0x52, 0x46, 0x51, 0x79, 0x51, 0x74,
                                                           0x30, 0x6d, 0x65, 0x53, 0x5a, 0x37, 0x6e, 0x34,
                                                           0x6e, 0x6b, 0x58, 0x45, 0x69, 0x4d, 0x62, 0x0a,
                                                           0x35, 0x6d, 0x55, 0x4a, 0x4f, 0x42, 0x62, 0x65,
                                                           0x69, 0x78, 0x2b, 0x6e, 0x4a, 0x66, 0x65, 0x39,
                                                           0x69, 0x6c, 0x34, 0x6a, 0x53, 0x5a, 0x4c, 0x34,
                                                           0x52, 0x63, 0x46, 0x44, 0x39, 0x65, 0x2b, 0x78,
                                                           0x35, 0x6e, 0x35, 0x41, 0x34, 0x77, 0x43, 0x52,
                                                           0x66, 0x63, 0x46, 0x52, 0x37, 0x6b, 0x36, 0x38,
                                                           0x59, 0x67, 0x43, 0x59, 0x45, 0x64, 0x48, 0x61,
                                                           0x4e, 0x5a, 0x46, 0x71, 0x45, 0x70, 0x33, 0x52,
                                                           0x0a, 0x34, 0x78, 0x6f, 0x74, 0x33, 0x53, 0x2f,
                                                           0x35, 0x63, 0x2f, 0x53, 0x63, 0x77, 0x31, 0x48,
                                                           0x61, 0x4b, 0x65, 0x36, 0x2f, 0x43, 0x64, 0x6f,
                                                           0x33, 0x79, 0x42, 0x58, 0x38, 0x6e, 0x53, 0x65,
                                                           0x6e, 0x38, 0x35, 0x46, 0x67, 0x70, 0x6b, 0x30,
                                                           0x56, 0x30, 0x32, 0x67, 0x43, 0x71, 0x65, 0x68,
                                                           0x70, 0x2b, 0x42, 0x41, 0x53, 0x2f, 0x42, 0x64,
                                                           0x59, 0x62, 0x6c, 0x36, 0x5a, 0x6b, 0x63, 0x59,
                                                           0x52, 0x0a, 0x43, 0x31, 0x67, 0x68, 0x64, 0x4e,
                                                           0x6b, 0x52, 0x59, 0x6f, 0x71, 0x59, 0x36, 0x77,
                                                           0x6f, 0x6f, 0x69, 0x74, 0x72, 0x2f, 0x4a, 0x57,
                                                           0x51, 0x6b, 0x72, 0x38, 0x45, 0x79, 0x65, 0x6b,
                                                           0x75, 0x35, 0x36, 0x2b, 0x79, 0x39, 0x63, 0x48,
                                                           0x42, 0x56, 0x67, 0x32, 0x44, 0x31, 0x6b, 0x61,
                                                           0x66, 0x6a, 0x4c, 0x2f, 0x57, 0x50, 0x69, 0x56,
                                                           0x6b, 0x53, 0x31, 0x47, 0x66, 0x47, 0x73, 0x62,
                                                           0x6a, 0x50, 0x0a, 0x39, 0x53, 0x53, 0x53, 0x72,
                                                           0x7a, 0x72, 0x71, 0x4c, 0x39, 0x32, 0x62, 0x35,
                                                           0x64, 0x4a, 0x37, 0x52, 0x4a, 0x76, 0x76, 0x62,
                                                           0x76, 0x5a, 0x74, 0x4a, 0x46, 0x64, 0x73, 0x45,
                                                           0x4c, 0x6b, 0x6f, 0x34, 0x51, 0x52, 0x7a, 0x7a,
                                                           0x61, 0x4d, 0x68, 0x67, 0x38, 0x64, 0x33, 0x73,
                                                           0x57, 0x33, 0x39, 0x2b, 0x62, 0x4e, 0x46, 0x31,
                                                           0x45, 0x48, 0x72, 0x62, 0x50, 0x4c, 0x6f, 0x57,
                                                           0x42, 0x5a, 0x7a, 0x0a, 0x6e, 0x49, 0x4f, 0x6d,
                                                           0x77, 0x64, 0x39, 0x4c, 0x73, 0x52, 0x31, 0x70,
                                                           0x74, 0x46, 0x75, 0x54, 0x33, 0x4f, 0x6a, 0x31,
                                                           0x39, 0x68, 0x4f, 0x4a, 0x45, 0x49, 0x6e, 0x65,
                                                           0x47, 0x59, 0x67, 0x4f, 0x68, 0x43, 0x7a, 0x6b,
                                                           0x6e, 0x78, 0x38, 0x47, 0x7a, 0x73, 0x68, 0x30,
                                                           0x77, 0x4c, 0x47, 0x36, 0x53, 0x45, 0x59, 0x63,
                                                           0x50, 0x72, 0x5a, 0x77, 0x71, 0x53, 0x58, 0x64,
                                                           0x34, 0x5a, 0x57, 0x43, 0x0a, 0x6a, 0x65, 0x48,
                                                           0x46, 0x48, 0x49, 0x45, 0x45, 0x4a, 0x7a, 0x68,
                                                           0x79, 0x30, 0x41, 0x66, 0x73, 0x68, 0x75, 0x42,
                                                           0x6b, 0x54, 0x74, 0x4a, 0x7a, 0x57, 0x49, 0x36,
                                                           0x35, 0x2b, 0x54, 0x50, 0x2f, 0x57, 0x73, 0x37,
                                                           0x49, 0x7a, 0x58, 0x6c, 0x6c, 0x37, 0x6f, 0x79,
                                                           0x63, 0x2b, 0x4e, 0x61, 0x36, 0x6d, 0x76, 0x79,
                                                           0x44, 0x33, 0x76, 0x70, 0x45, 0x77, 0x7a, 0x6a,
                                                           0x7a, 0x35, 0x75, 0x6d, 0x52, 0x0a, 0x44, 0x32,
                                                           0x4d, 0x56, 0x48, 0x6d, 0x52, 0x78, 0x5a, 0x36,
                                                           0x4a, 0x57, 0x6d, 0x6d, 0x61, 0x52, 0x53, 0x79,
                                                           0x4b, 0x6d, 0x44, 0x32, 0x45, 0x33, 0x74, 0x5a,
                                                           0x72, 0x66, 0x59, 0x31, 0x33, 0x62, 0x75, 0x38,
                                                           0x6d, 0x65, 0x4c, 0x54, 0x32, 0x7a, 0x50, 0x4f,
                                                           0x5a, 0x6a, 0x6f, 0x50, 0x64, 0x38, 0x74, 0x66,
                                                           0x72, 0x6e, 0x41, 0x48, 0x54, 0x66, 0x52, 0x30,
                                                           0x31, 0x51, 0x55, 0x77, 0x62, 0x39, 0x0a, 0x67,
                                                           0x34, 0x64, 0x75, 0x45, 0x31, 0x74, 0x70, 0x61,
                                                           0x71, 0x77, 0x62, 0x2f, 0x49, 0x47, 0x43, 0x4f,
                                                           0x2f, 0x34, 0x45, 0x76, 0x72, 0x63, 0x34, 0x61,
                                                           0x58, 0x31, 0x63, 0x75, 0x59, 0x79, 0x55, 0x48,
                                                           0x55, 0x6b, 0x55, 0x32, 0x2b, 0x73, 0x75, 0x56,
                                                           0x32, 0x45, 0x4d, 0x4d, 0x6a, 0x39, 0x37, 0x2b,
                                                           0x6c, 0x7a, 0x47, 0x77, 0x64, 0x61, 0x52, 0x38,
                                                           0x62, 0x78, 0x4b, 0x42, 0x6f, 0x6a, 0x6b, 0x0a,
                                                           0x55, 0x42, 0x38, 0x74, 0x32, 0x74, 0x52, 0x35,
                                                           0x49, 0x43, 0x66, 0x45, 0x63, 0x34, 0x42, 0x2b,
                                                           0x4d, 0x5a, 0x6e, 0x6e, 0x6b, 0x31, 0x43, 0x53,
                                                           0x67, 0x41, 0x79, 0x48, 0x4d, 0x6a, 0x33, 0x6a,
                                                           0x71, 0x76, 0x41, 0x37, 0x78, 0x6a, 0x36, 0x32,
                                                           0x4f, 0x36, 0x32, 0x5a, 0x38, 0x34, 0x70, 0x64,
                                                           0x38, 0x39, 0x43, 0x48, 0x46, 0x70, 0x4a, 0x55,
                                                           0x55, 0x35, 0x65, 0x62, 0x79, 0x6f, 0x35, 0x76,
                                                           0x0a, 0x71, 0x67, 0x72, 0x65, 0x2b, 0x47, 0x62,
                                                           0x46, 0x68, 0x4b, 0x38, 0x50, 0x50, 0x41, 0x6f,
                                                           0x64, 0x51, 0x2f, 0x77, 0x48, 0x45, 0x65, 0x68,
                                                           0x41, 0x57, 0x47, 0x42, 0x45, 0x68, 0x52, 0x4d,
                                                           0x35, 0x5a, 0x7a, 0x6e, 0x47, 0x78, 0x46, 0x6b,
                                                           0x69, 0x58, 0x72, 0x44, 0x30, 0x4f, 0x4e, 0x4f,
                                                           0x4f, 0x37, 0x48, 0x4b, 0x44, 0x63, 0x31, 0x6e,
                                                           0x5a, 0x73, 0x4d, 0x4b, 0x57, 0x68, 0x75, 0x68,
                                                           0x6a, 0x0a, 0x7a, 0x34, 0x54, 0x67, 0x4b, 0x62,
                                                           0x41, 0x65, 0x64, 0x47, 0x73, 0x74, 0x78, 0x70,
                                                           0x68, 0x75, 0x69, 0x4f, 0x73, 0x57, 0x5a, 0x58,
                                                           0x73, 0x57, 0x61, 0x39, 0x42, 0x74, 0x75, 0x6b,
                                                           0x48, 0x41, 0x61, 0x63, 0x75, 0x59, 0x58, 0x62,
                                                           0x49, 0x2f, 0x33, 0x63, 0x54, 0x6a, 0x43, 0x2b,
                                                           0x4e, 0x4d, 0x35, 0x6a, 0x36, 0x61, 0x33, 0x75,
                                                           0x7a, 0x78, 0x46, 0x35, 0x56, 0x4b, 0x61, 0x63,
                                                           0x30, 0x2b, 0x0a, 0x72, 0x2b, 0x56, 0x35, 0x74,
                                                           0x48, 0x34, 0x38, 0x4f, 0x36, 0x73, 0x2f, 0x77,
                                                           0x4e, 0x72, 0x72, 0x57, 0x34, 0x63, 0x50, 0x6e,
                                                           0x31, 0x64, 0x42, 0x39, 0x6e, 0x50, 0x4e, 0x48,
                                                           0x64, 0x61, 0x5a, 0x65, 0x6a, 0x55, 0x35, 0x6f,
                                                           0x5a, 0x56, 0x4c, 0x4f, 0x66, 0x69, 0x56, 0x44,
                                                           0x30, 0x57, 0x74, 0x38, 0x67, 0x74, 0x76, 0x5a,
                                                           0x4a, 0x65, 0x4d, 0x74, 0x71, 0x77, 0x42, 0x36,
                                                           0x31, 0x74, 0x6a, 0x0a, 0x52, 0x70, 0x78, 0x76,
                                                           0x6a, 0x37, 0x59, 0x3d, 0x0a, 0x2d, 0x2d, 0x2d,
                                                           0x2d, 0x2d, 0x45, 0x4e, 0x44, 0x20, 0x45, 0x4e,
                                                           0x43, 0x52, 0x59, 0x50, 0x54, 0x45, 0x44, 0x20,
                                                           0x50, 0x52, 0x49, 0x56, 0x41, 0x54, 0x45, 0x20,
                                                           0x4b, 0x45, 0x59, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d,
                                                           0x0a
                                                          };

// Number of items in the gUSecurityCredentialTestKey1024Pkcs8Pem array.
//lint -esym(552, gUSecurityCredentialTestKey1024Pkcs8PemSize)
// Suppress variable not referenced.
const size_t gUSecurityCredentialTestKey1024Pkcs8PemSize =
    sizeof(gUSecurityCredentialTestKey1024Pkcs8Pem) /
    sizeof(gUSecurityCredentialTestKey1024Pkcs8Pem[0]);

/* ----------------------------------------------------------------
 * VARIABLES: CERTIFICATES
 * -------------------------------------------------------------- */

// A test root/CA X.509 PEM-format certificate generated on
// 2021/02/10 16:52:35 UTC with the 1024-bit key using OpenSSL as
// follows:
//
// 1. Create CA certificate:  openssl req -x509 -new -nodes -key key_1024_pkcs1.pem -sha256 -days 1825 -out cert_ca_x509.pem
//                            ...with non-default fields as follows:
//                            Country Name:      GB
//                            Locality Name:     Cambridge
//                            Organization Name: u-blox
//                            Common Name:       ubxlib ca
//    Note: 1825 days is 5 years, so expiration is 2026/02/09 16:52:35 or 1770655955.
//
// See cert_ca_x509.pem.
//lint -esym(552, gUSecurityCredentialTestRootCaX509Pem) Suppress
// variable not referenced.
const uint8_t gUSecurityCredentialTestRootCaX509Pem[] = {0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x42, 0x45, 0x47,
                                                         0x49, 0x4e, 0x20, 0x43, 0x45, 0x52, 0x54, 0x49,
                                                         0x46, 0x49, 0x43, 0x41, 0x54, 0x45, 0x2d, 0x2d,
                                                         0x2d, 0x2d, 0x2d, 0x0d, 0x0a, 0x4d, 0x49, 0x49,
                                                         0x43, 0x68, 0x44, 0x43, 0x43, 0x41, 0x65, 0x32,
                                                         0x67, 0x41, 0x77, 0x49, 0x42, 0x41, 0x67, 0x49,
                                                         0x4a, 0x41, 0x4d, 0x51, 0x76, 0x42, 0x61, 0x4b,
                                                         0x34, 0x68, 0x6b, 0x73, 0x76, 0x4d, 0x41, 0x30,
                                                         0x47, 0x43, 0x53, 0x71, 0x47, 0x53, 0x49, 0x62,
                                                         0x33, 0x44, 0x51, 0x45, 0x42, 0x43, 0x77, 0x55,
                                                         0x41, 0x4d, 0x46, 0x73, 0x78, 0x43, 0x7a, 0x41,
                                                         0x4a, 0x42, 0x67, 0x4e, 0x56, 0x0d, 0x0a, 0x42,
                                                         0x41, 0x59, 0x54, 0x41, 0x6b, 0x64, 0x43, 0x4d,
                                                         0x52, 0x4d, 0x77, 0x45, 0x51, 0x59, 0x44, 0x56,
                                                         0x51, 0x51, 0x49, 0x44, 0x41, 0x70, 0x54, 0x62,
                                                         0x32, 0x31, 0x6c, 0x4c, 0x56, 0x4e, 0x30, 0x59,
                                                         0x58, 0x52, 0x6c, 0x4d, 0x52, 0x49, 0x77, 0x45,
                                                         0x41, 0x59, 0x44, 0x56, 0x51, 0x51, 0x48, 0x44,
                                                         0x41, 0x6c, 0x44, 0x59, 0x57, 0x31, 0x69, 0x63,
                                                         0x6d, 0x6c, 0x6b, 0x5a, 0x32, 0x55, 0x78, 0x0d,
                                                         0x0a, 0x44, 0x7a, 0x41, 0x4e, 0x42, 0x67, 0x4e,
                                                         0x56, 0x42, 0x41, 0x6f, 0x4d, 0x42, 0x6e, 0x55,
                                                         0x74, 0x59, 0x6d, 0x78, 0x76, 0x65, 0x44, 0x45,
                                                         0x53, 0x4d, 0x42, 0x41, 0x47, 0x41, 0x31, 0x55,
                                                         0x45, 0x41, 0x77, 0x77, 0x4a, 0x64, 0x57, 0x4a,
                                                         0x34, 0x62, 0x47, 0x6c, 0x69, 0x49, 0x47, 0x4e,
                                                         0x68, 0x4d, 0x42, 0x34, 0x58, 0x44, 0x54, 0x49,
                                                         0x78, 0x4d, 0x44, 0x49, 0x78, 0x4d, 0x44, 0x45,
                                                         0x32, 0x0d, 0x0a, 0x4e, 0x54, 0x49, 0x7a, 0x4e,
                                                         0x56, 0x6f, 0x58, 0x44, 0x54, 0x49, 0x32, 0x4d,
                                                         0x44, 0x49, 0x77, 0x4f, 0x54, 0x45, 0x32, 0x4e,
                                                         0x54, 0x49, 0x7a, 0x4e, 0x56, 0x6f, 0x77, 0x57,
                                                         0x7a, 0x45, 0x4c, 0x4d, 0x41, 0x6b, 0x47, 0x41,
                                                         0x31, 0x55, 0x45, 0x42, 0x68, 0x4d, 0x43, 0x52,
                                                         0x30, 0x49, 0x78, 0x45, 0x7a, 0x41, 0x52, 0x42,
                                                         0x67, 0x4e, 0x56, 0x42, 0x41, 0x67, 0x4d, 0x43,
                                                         0x6c, 0x4e, 0x76, 0x0d, 0x0a, 0x62, 0x57, 0x55,
                                                         0x74, 0x55, 0x33, 0x52, 0x68, 0x64, 0x47, 0x55,
                                                         0x78, 0x45, 0x6a, 0x41, 0x51, 0x42, 0x67, 0x4e,
                                                         0x56, 0x42, 0x41, 0x63, 0x4d, 0x43, 0x55, 0x4e,
                                                         0x68, 0x62, 0x57, 0x4a, 0x79, 0x61, 0x57, 0x52,
                                                         0x6e, 0x5a, 0x54, 0x45, 0x50, 0x4d, 0x41, 0x30,
                                                         0x47, 0x41, 0x31, 0x55, 0x45, 0x43, 0x67, 0x77,
                                                         0x47, 0x64, 0x53, 0x31, 0x69, 0x62, 0x47, 0x39,
                                                         0x34, 0x4d, 0x52, 0x49, 0x77, 0x0d, 0x0a, 0x45,
                                                         0x41, 0x59, 0x44, 0x56, 0x51, 0x51, 0x44, 0x44,
                                                         0x41, 0x6c, 0x31, 0x59, 0x6e, 0x68, 0x73, 0x61,
                                                         0x57, 0x49, 0x67, 0x59, 0x32, 0x45, 0x77, 0x67,
                                                         0x5a, 0x38, 0x77, 0x44, 0x51, 0x59, 0x4a, 0x4b,
                                                         0x6f, 0x5a, 0x49, 0x68, 0x76, 0x63, 0x4e, 0x41,
                                                         0x51, 0x45, 0x42, 0x42, 0x51, 0x41, 0x44, 0x67,
                                                         0x59, 0x30, 0x41, 0x4d, 0x49, 0x47, 0x4a, 0x41,
                                                         0x6f, 0x47, 0x42, 0x41, 0x4c, 0x43, 0x66, 0x0d,
                                                         0x0a, 0x46, 0x51, 0x36, 0x4f, 0x73, 0x6d, 0x30,
                                                         0x58, 0x6a, 0x73, 0x73, 0x44, 0x70, 0x68, 0x2b,
                                                         0x39, 0x4e, 0x30, 0x51, 0x4b, 0x7a, 0x76, 0x77,
                                                         0x51, 0x5a, 0x45, 0x56, 0x6a, 0x54, 0x69, 0x64,
                                                         0x6b, 0x68, 0x31, 0x2b, 0x48, 0x70, 0x42, 0x53,
                                                         0x45, 0x45, 0x62, 0x33, 0x61, 0x36, 0x4b, 0x7a,
                                                         0x66, 0x4c, 0x48, 0x4c, 0x58, 0x57, 0x50, 0x7a,
                                                         0x46, 0x46, 0x56, 0x56, 0x77, 0x6c, 0x6a, 0x75,
                                                         0x51, 0x0d, 0x0a, 0x73, 0x41, 0x77, 0x72, 0x4c,
                                                         0x42, 0x59, 0x76, 0x52, 0x54, 0x59, 0x69, 0x31,
                                                         0x76, 0x6a, 0x45, 0x44, 0x44, 0x32, 0x45, 0x33,
                                                         0x75, 0x58, 0x50, 0x79, 0x47, 0x63, 0x73, 0x73,
                                                         0x36, 0x41, 0x6b, 0x68, 0x39, 0x6d, 0x54, 0x52,
                                                         0x36, 0x55, 0x47, 0x53, 0x36, 0x48, 0x77, 0x32,
                                                         0x48, 0x46, 0x6c, 0x48, 0x33, 0x50, 0x57, 0x44,
                                                         0x39, 0x34, 0x4c, 0x52, 0x6a, 0x69, 0x52, 0x44,
                                                         0x77, 0x4e, 0x6e, 0x0d, 0x0a, 0x36, 0x43, 0x34,
                                                         0x68, 0x70, 0x72, 0x61, 0x6a, 0x30, 0x70, 0x5a,
                                                         0x52, 0x62, 0x49, 0x42, 0x45, 0x37, 0x75, 0x57,
                                                         0x46, 0x66, 0x4e, 0x42, 0x39, 0x71, 0x38, 0x46,
                                                         0x6a, 0x52, 0x38, 0x76, 0x35, 0x70, 0x6f, 0x7a,
                                                         0x4c, 0x50, 0x69, 0x5a, 0x56, 0x41, 0x67, 0x4d,
                                                         0x42, 0x41, 0x41, 0x47, 0x6a, 0x55, 0x44, 0x42,
                                                         0x4f, 0x4d, 0x42, 0x30, 0x47, 0x41, 0x31, 0x55,
                                                         0x64, 0x44, 0x67, 0x51, 0x57, 0x0d, 0x0a, 0x42,
                                                         0x42, 0x53, 0x35, 0x59, 0x6c, 0x2b, 0x67, 0x51,
                                                         0x57, 0x36, 0x4d, 0x63, 0x6a, 0x67, 0x59, 0x75,
                                                         0x47, 0x79, 0x41, 0x44, 0x37, 0x61, 0x62, 0x39,
                                                         0x63, 0x76, 0x4d, 0x6e, 0x54, 0x41, 0x66, 0x42,
                                                         0x67, 0x4e, 0x56, 0x48, 0x53, 0x4d, 0x45, 0x47,
                                                         0x44, 0x41, 0x57, 0x67, 0x42, 0x53, 0x35, 0x59,
                                                         0x6c, 0x2b, 0x67, 0x51, 0x57, 0x36, 0x4d, 0x63,
                                                         0x6a, 0x67, 0x59, 0x75, 0x47, 0x79, 0x41, 0x0d,
                                                         0x0a, 0x44, 0x37, 0x61, 0x62, 0x39, 0x63, 0x76,
                                                         0x4d, 0x6e, 0x54, 0x41, 0x4d, 0x42, 0x67, 0x4e,
                                                         0x56, 0x48, 0x52, 0x4d, 0x45, 0x42, 0x54, 0x41,
                                                         0x44, 0x41, 0x51, 0x48, 0x2f, 0x4d, 0x41, 0x30,
                                                         0x47, 0x43, 0x53, 0x71, 0x47, 0x53, 0x49, 0x62,
                                                         0x33, 0x44, 0x51, 0x45, 0x42, 0x43, 0x77, 0x55,
                                                         0x41, 0x41, 0x34, 0x47, 0x42, 0x41, 0x4b, 0x6a,
                                                         0x31, 0x36, 0x34, 0x69, 0x2b, 0x6c, 0x65, 0x4c,
                                                         0x69, 0x0d, 0x0a, 0x34, 0x6e, 0x6a, 0x5a, 0x6e,
                                                         0x55, 0x35, 0x33, 0x4d, 0x37, 0x50, 0x69, 0x42,
                                                         0x45, 0x6f, 0x63, 0x58, 0x59, 0x37, 0x2f, 0x49,
                                                         0x69, 0x6c, 0x70, 0x31, 0x51, 0x4f, 0x77, 0x32,
                                                         0x47, 0x69, 0x66, 0x45, 0x31, 0x55, 0x37, 0x6a,
                                                         0x67, 0x64, 0x52, 0x65, 0x6a, 0x2b, 0x37, 0x44,
                                                         0x4b, 0x4b, 0x79, 0x4c, 0x78, 0x63, 0x32, 0x4c,
                                                         0x35, 0x69, 0x45, 0x34, 0x48, 0x4c, 0x56, 0x4d,
                                                         0x51, 0x70, 0x45, 0x0d, 0x0a, 0x4b, 0x79, 0x76,
                                                         0x6b, 0x45, 0x6d, 0x30, 0x7a, 0x43, 0x4f, 0x68,
                                                         0x6b, 0x48, 0x50, 0x74, 0x6a, 0x64, 0x44, 0x73,
                                                         0x62, 0x38, 0x66, 0x58, 0x65, 0x59, 0x59, 0x44,
                                                         0x6e, 0x75, 0x42, 0x6d, 0x75, 0x75, 0x75, 0x43,
                                                         0x51, 0x76, 0x79, 0x49, 0x46, 0x41, 0x66, 0x4a,
                                                         0x4c, 0x4f, 0x61, 0x76, 0x42, 0x32, 0x66, 0x46,
                                                         0x62, 0x79, 0x42, 0x39, 0x67, 0x46, 0x78, 0x6b,
                                                         0x63, 0x4a, 0x41, 0x4e, 0x6f, 0x0d, 0x0a, 0x2b,
                                                         0x68, 0x38, 0x7a, 0x48, 0x4a, 0x6c, 0x75, 0x44,
                                                         0x56, 0x79, 0x4b, 0x64, 0x45, 0x2b, 0x59, 0x37,
                                                         0x58, 0x6b, 0x47, 0x36, 0x6d, 0x66, 0x6f, 0x36,
                                                         0x65, 0x2f, 0x35, 0x47, 0x38, 0x75, 0x54, 0x0d,
                                                         0x0a, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x45, 0x4e,
                                                         0x44, 0x20, 0x43, 0x45, 0x52, 0x54, 0x49, 0x46,
                                                         0x49, 0x43, 0x41, 0x54, 0x45, 0x2d, 0x2d, 0x2d,
                                                         0x2d, 0x2d, 0x0d, 0x0a
                                                        };

// Number of items in the gUSecurityCredentialTestRootCaX509Pem array.
//lint -esym(552, gUSecurityCredentialTestRootCaX509PemSize) Suppress
// variable not referenced.
const size_t gUSecurityCredentialTestRootCaX509PemSize =
    sizeof(gUSecurityCredentialTestRootCaX509Pem) /
    sizeof(gUSecurityCredentialTestRootCaX509Pem[0]);

// A test client X.509 PEM-format certificate generated on
// 2021/02/10 17:19:08 UTC with the above root certificate and
// the same 1024-bit key (to keep things simple for testing)
// using OpenSSL as follows:
//
// 1. Create certificate signing request:      openssl req -new -nodes -key cert_ca_x509.pem -sha256 -days 1825 -out cert_client_x509_req.pem
//                                             ...with passphrase "ubxlib" and non-default fields as follows:
//                                             Country Name:      GB
//                                             Locality Name:     Cambridge
//                                             Organization Name: u-blox
//                                             Common Name:       ubxlib client
// 2. Create signed certificate using root CA: openssl x509 -req -in cert_client_x509_req.pem -sha256 -days 1825 -CA cert_ca_x509.pem -CAkey key_1024_pkcs1.pem -set_serial 01 -out cert_client_x509.pem
//                                             ...with passphrase "ubxlib"
//    Note: 1825 days is 5 years, so expiration is 2026/02/09 17:19:08 or 1770657548.
//
// See cert_client_x509.pem.
const uint8_t gUSecurityCredentialTestClientX509Pem[] = {0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x42, 0x45, 0x47,
                                                         0x49, 0x4e, 0x20, 0x43, 0x45, 0x52, 0x54, 0x49,
                                                         0x46, 0x49, 0x43, 0x41, 0x54, 0x45, 0x2d, 0x2d,
                                                         0x2d, 0x2d, 0x2d, 0x0d, 0x0a, 0x4d, 0x49, 0x49,
                                                         0x43, 0x4b, 0x54, 0x43, 0x43, 0x41, 0x5a, 0x49,
                                                         0x43, 0x41, 0x51, 0x45, 0x77, 0x44, 0x51, 0x59,
                                                         0x4a, 0x4b, 0x6f, 0x5a, 0x49, 0x68, 0x76, 0x63,
                                                         0x4e, 0x41, 0x51, 0x45, 0x4c, 0x42, 0x51, 0x41,
                                                         0x77, 0x57, 0x7a, 0x45, 0x4c, 0x4d, 0x41, 0x6b,
                                                         0x47, 0x41, 0x31, 0x55, 0x45, 0x42, 0x68, 0x4d,
                                                         0x43, 0x52, 0x30, 0x49, 0x78, 0x45, 0x7a, 0x41,
                                                         0x52, 0x42, 0x67, 0x4e, 0x56, 0x0d, 0x0a, 0x42,
                                                         0x41, 0x67, 0x4d, 0x43, 0x6c, 0x4e, 0x76, 0x62,
                                                         0x57, 0x55, 0x74, 0x55, 0x33, 0x52, 0x68, 0x64,
                                                         0x47, 0x55, 0x78, 0x45, 0x6a, 0x41, 0x51, 0x42,
                                                         0x67, 0x4e, 0x56, 0x42, 0x41, 0x63, 0x4d, 0x43,
                                                         0x55, 0x4e, 0x68, 0x62, 0x57, 0x4a, 0x79, 0x61,
                                                         0x57, 0x52, 0x6e, 0x5a, 0x54, 0x45, 0x50, 0x4d,
                                                         0x41, 0x30, 0x47, 0x41, 0x31, 0x55, 0x45, 0x43,
                                                         0x67, 0x77, 0x47, 0x64, 0x53, 0x31, 0x69, 0x0d,
                                                         0x0a, 0x62, 0x47, 0x39, 0x34, 0x4d, 0x52, 0x49,
                                                         0x77, 0x45, 0x41, 0x59, 0x44, 0x56, 0x51, 0x51,
                                                         0x44, 0x44, 0x41, 0x6c, 0x31, 0x59, 0x6e, 0x68,
                                                         0x73, 0x61, 0x57, 0x49, 0x67, 0x59, 0x32, 0x45,
                                                         0x77, 0x48, 0x68, 0x63, 0x4e, 0x4d, 0x6a, 0x45,
                                                         0x77, 0x4d, 0x6a, 0x45, 0x77, 0x4d, 0x54, 0x63,
                                                         0x78, 0x4f, 0x54, 0x41, 0x34, 0x57, 0x68, 0x63,
                                                         0x4e, 0x4d, 0x6a, 0x59, 0x77, 0x4d, 0x6a, 0x41,
                                                         0x35, 0x0d, 0x0a, 0x4d, 0x54, 0x63, 0x78, 0x4f,
                                                         0x54, 0x41, 0x34, 0x57, 0x6a, 0x42, 0x66, 0x4d,
                                                         0x51, 0x73, 0x77, 0x43, 0x51, 0x59, 0x44, 0x56,
                                                         0x51, 0x51, 0x47, 0x45, 0x77, 0x4a, 0x48, 0x51,
                                                         0x6a, 0x45, 0x54, 0x4d, 0x42, 0x45, 0x47, 0x41,
                                                         0x31, 0x55, 0x45, 0x43, 0x41, 0x77, 0x4b, 0x55,
                                                         0x32, 0x39, 0x74, 0x5a, 0x53, 0x31, 0x54, 0x64,
                                                         0x47, 0x46, 0x30, 0x5a, 0x54, 0x45, 0x53, 0x4d,
                                                         0x42, 0x41, 0x47, 0x0d, 0x0a, 0x41, 0x31, 0x55,
                                                         0x45, 0x42, 0x77, 0x77, 0x4a, 0x51, 0x32, 0x46,
                                                         0x74, 0x59, 0x6e, 0x4a, 0x70, 0x5a, 0x47, 0x64,
                                                         0x6c, 0x4d, 0x51, 0x38, 0x77, 0x44, 0x51, 0x59,
                                                         0x44, 0x56, 0x51, 0x51, 0x4b, 0x44, 0x41, 0x5a,
                                                         0x31, 0x4c, 0x57, 0x4a, 0x73, 0x62, 0x33, 0x67,
                                                         0x78, 0x46, 0x6a, 0x41, 0x55, 0x42, 0x67, 0x4e,
                                                         0x56, 0x42, 0x41, 0x4d, 0x4d, 0x44, 0x58, 0x56,
                                                         0x69, 0x65, 0x47, 0x78, 0x70, 0x0d, 0x0a, 0x59,
                                                         0x69, 0x42, 0x6a, 0x62, 0x47, 0x6c, 0x6c, 0x62,
                                                         0x6e, 0x51, 0x77, 0x67, 0x5a, 0x38, 0x77, 0x44,
                                                         0x51, 0x59, 0x4a, 0x4b, 0x6f, 0x5a, 0x49, 0x68,
                                                         0x76, 0x63, 0x4e, 0x41, 0x51, 0x45, 0x42, 0x42,
                                                         0x51, 0x41, 0x44, 0x67, 0x59, 0x30, 0x41, 0x4d,
                                                         0x49, 0x47, 0x4a, 0x41, 0x6f, 0x47, 0x42, 0x41,
                                                         0x4c, 0x43, 0x66, 0x46, 0x51, 0x36, 0x4f, 0x73,
                                                         0x6d, 0x30, 0x58, 0x6a, 0x73, 0x73, 0x44, 0x0d,
                                                         0x0a, 0x70, 0x68, 0x2b, 0x39, 0x4e, 0x30, 0x51,
                                                         0x4b, 0x7a, 0x76, 0x77, 0x51, 0x5a, 0x45, 0x56,
                                                         0x6a, 0x54, 0x69, 0x64, 0x6b, 0x68, 0x31, 0x2b,
                                                         0x48, 0x70, 0x42, 0x53, 0x45, 0x45, 0x62, 0x33,
                                                         0x61, 0x36, 0x4b, 0x7a, 0x66, 0x4c, 0x48, 0x4c,
                                                         0x58, 0x57, 0x50, 0x7a, 0x46, 0x46, 0x56, 0x56,
                                                         0x77, 0x6c, 0x6a, 0x75, 0x51, 0x73, 0x41, 0x77,
                                                         0x72, 0x4c, 0x42, 0x59, 0x76, 0x52, 0x54, 0x59,
                                                         0x69, 0x0d, 0x0a, 0x31, 0x76, 0x6a, 0x45, 0x44,
                                                         0x44, 0x32, 0x45, 0x33, 0x75, 0x58, 0x50, 0x79,
                                                         0x47, 0x63, 0x73, 0x73, 0x36, 0x41, 0x6b, 0x68,
                                                         0x39, 0x6d, 0x54, 0x52, 0x36, 0x55, 0x47, 0x53,
                                                         0x36, 0x48, 0x77, 0x32, 0x48, 0x46, 0x6c, 0x48,
                                                         0x33, 0x50, 0x57, 0x44, 0x39, 0x34, 0x4c, 0x52,
                                                         0x6a, 0x69, 0x52, 0x44, 0x77, 0x4e, 0x6e, 0x36,
                                                         0x43, 0x34, 0x68, 0x70, 0x72, 0x61, 0x6a, 0x30,
                                                         0x70, 0x5a, 0x52, 0x0d, 0x0a, 0x62, 0x49, 0x42,
                                                         0x45, 0x37, 0x75, 0x57, 0x46, 0x66, 0x4e, 0x42,
                                                         0x39, 0x71, 0x38, 0x46, 0x6a, 0x52, 0x38, 0x76,
                                                         0x35, 0x70, 0x6f, 0x7a, 0x4c, 0x50, 0x69, 0x5a,
                                                         0x56, 0x41, 0x67, 0x4d, 0x42, 0x41, 0x41, 0x45,
                                                         0x77, 0x44, 0x51, 0x59, 0x4a, 0x4b, 0x6f, 0x5a,
                                                         0x49, 0x68, 0x76, 0x63, 0x4e, 0x41, 0x51, 0x45,
                                                         0x4c, 0x42, 0x51, 0x41, 0x44, 0x67, 0x59, 0x45,
                                                         0x41, 0x48, 0x77, 0x31, 0x56, 0x0d, 0x0a, 0x63,
                                                         0x49, 0x51, 0x64, 0x4a, 0x52, 0x36, 0x45, 0x45,
                                                         0x54, 0x32, 0x5a, 0x77, 0x32, 0x37, 0x55, 0x79,
                                                         0x53, 0x6c, 0x41, 0x32, 0x77, 0x57, 0x45, 0x34,
                                                         0x47, 0x38, 0x56, 0x41, 0x4c, 0x31, 0x46, 0x69,
                                                         0x48, 0x67, 0x6a, 0x41, 0x59, 0x42, 0x42, 0x78,
                                                         0x61, 0x31, 0x7a, 0x61, 0x6e, 0x41, 0x61, 0x57,
                                                         0x65, 0x6a, 0x36, 0x68, 0x56, 0x58, 0x6e, 0x31,
                                                         0x71, 0x68, 0x53, 0x32, 0x34, 0x4b, 0x4e, 0x0d,
                                                         0x0a, 0x45, 0x76, 0x6f, 0x34, 0x51, 0x34, 0x42,
                                                         0x71, 0x71, 0x70, 0x65, 0x73, 0x33, 0x6c, 0x72,
                                                         0x36, 0x75, 0x2f, 0x54, 0x58, 0x51, 0x52, 0x67,
                                                         0x54, 0x68, 0x77, 0x53, 0x6e, 0x4a, 0x56, 0x4d,
                                                         0x4c, 0x74, 0x61, 0x6a, 0x67, 0x53, 0x4a, 0x66,
                                                         0x73, 0x71, 0x34, 0x6c, 0x56, 0x4d, 0x4f, 0x59,
                                                         0x75, 0x5a, 0x52, 0x50, 0x71, 0x69, 0x48, 0x47,
                                                         0x64, 0x4a, 0x63, 0x69, 0x6c, 0x61, 0x4b, 0x37,
                                                         0x54, 0x0d, 0x0a, 0x6a, 0x34, 0x55, 0x4e, 0x78,
                                                         0x67, 0x68, 0x79, 0x79, 0x49, 0x44, 0x58, 0x46,
                                                         0x69, 0x79, 0x56, 0x30, 0x45, 0x5a, 0x42, 0x57,
                                                         0x31, 0x72, 0x75, 0x6c, 0x61, 0x4c, 0x6c, 0x34,
                                                         0x64, 0x73, 0x2b, 0x70, 0x61, 0x47, 0x4e, 0x38,
                                                         0x69, 0x45, 0x3d, 0x0d, 0x0a, 0x2d, 0x2d, 0x2d,
                                                         0x2d, 0x2d, 0x45, 0x4e, 0x44, 0x20, 0x43, 0x45,
                                                         0x52, 0x54, 0x49, 0x46, 0x49, 0x43, 0x41, 0x54,
                                                         0x45, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x0d, 0x0a
                                                        };

// Number of items in the gUSecurityCredentialTestClientX509Pem array.
const size_t gUSecurityCredentialTestClientX509PemSize =
    sizeof(gUSecurityCredentialTestClientX509Pem) /
    sizeof(gUSecurityCredentialTestClientX509Pem[0]);

#ifdef U_SECURITY_CREDENTIAL_TEST_FORMATS

/* ----------------------------------------------------------------
 * VARIABLES: FORMAT TEST DATA
 * -------------------------------------------------------------- */

/** A table of all possible formats/encodings of credentials,
 * used to explore what is supported by a given module.
 */
const uSecurityCredentialFormatTest_t gUSecurityCredentialTestFormat[] = {
    {
        "1024 bit PEM/PKCS1", U_SECURITY_CREDENTIAL_CLIENT_KEY_PRIVATE,
        887, NULL,
        //lint -e785 Suppress Too few initializers
        {
            0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x42, 0x45, 0x47, 0x49, 0x4e, 0x20, 0x52, 0x53, 0x41, 0x20, 0x50,
            0x52, 0x49, 0x56, 0x41, 0x54, 0x45, 0x20, 0x4b, 0x45, 0x59, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x0a,
            0x4d, 0x49, 0x49, 0x43, 0x57, 0x77, 0x49, 0x42, 0x41, 0x41, 0x4b, 0x42, 0x67, 0x51, 0x43, 0x77,
            0x6e, 0x78, 0x55, 0x4f, 0x6a, 0x72, 0x4a, 0x74, 0x46, 0x34, 0x37, 0x4c, 0x41, 0x36, 0x59, 0x66,
            0x76, 0x54, 0x64, 0x45, 0x43, 0x73, 0x37, 0x38, 0x45, 0x47, 0x52, 0x46, 0x59, 0x30, 0x34, 0x6e,
            0x5a, 0x49, 0x64, 0x66, 0x68, 0x36, 0x51, 0x55, 0x68, 0x42, 0x47, 0x39, 0x32, 0x75, 0x69, 0x73,
            0x0a, 0x33, 0x79, 0x78, 0x79, 0x31, 0x31, 0x6a, 0x38, 0x78, 0x52, 0x56, 0x56, 0x63, 0x4a, 0x59,
            0x37, 0x6b, 0x4c, 0x41, 0x4d, 0x4b, 0x79, 0x77, 0x57, 0x4c, 0x30, 0x55, 0x32, 0x49, 0x74, 0x62,
            0x34, 0x78, 0x41, 0x77, 0x39, 0x68, 0x4e, 0x37, 0x6c, 0x7a, 0x38, 0x68, 0x6e, 0x4c, 0x4c, 0x4f,
            0x67, 0x4a, 0x49, 0x66, 0x5a, 0x6b, 0x30, 0x65, 0x6c, 0x42, 0x6b, 0x75, 0x68, 0x38, 0x4e, 0x68,
            0x78, 0x0a, 0x5a, 0x52, 0x39, 0x7a, 0x31, 0x67, 0x2f, 0x65, 0x43, 0x30, 0x59, 0x34, 0x6b, 0x51,
            0x38, 0x44, 0x5a, 0x2b, 0x67, 0x75, 0x49, 0x61, 0x61, 0x32, 0x6f, 0x39, 0x4b, 0x57, 0x55, 0x57,
            0x79, 0x41, 0x52, 0x4f, 0x37, 0x6c, 0x68, 0x58, 0x7a, 0x51, 0x66, 0x61, 0x76, 0x42, 0x59, 0x30,
            0x66, 0x4c, 0x2b, 0x61, 0x61, 0x4d, 0x79, 0x7a, 0x34, 0x6d, 0x56, 0x51, 0x49, 0x44, 0x41, 0x51,
            0x41, 0x42, 0x0a, 0x41, 0x6f, 0x47, 0x41, 0x41, 0x38, 0x42, 0x64, 0x51, 0x69, 0x48, 0x6c, 0x79,
            0x45, 0x74, 0x6e, 0x7a, 0x51, 0x67, 0x57, 0x7a, 0x7a, 0x58, 0x46, 0x70, 0x72, 0x78, 0x55, 0x43,
            0x53, 0x61, 0x50, 0x2b, 0x71, 0x6c, 0x52, 0x54, 0x36, 0x54, 0x69, 0x2b, 0x7a, 0x52, 0x32, 0x56,
            0x37, 0x6b, 0x32, 0x5a, 0x30, 0x48, 0x76, 0x44, 0x7a, 0x7a, 0x75, 0x52, 0x4d, 0x49, 0x75, 0x4d,
            0x62, 0x41, 0x39, 0x0a, 0x38, 0x38, 0x4b, 0x67, 0x62, 0x6b, 0x4e, 0x30, 0x65, 0x63, 0x32, 0x77,
            0x38, 0x67, 0x65, 0x76, 0x41, 0x4b, 0x64, 0x63, 0x79, 0x59, 0x76, 0x67, 0x64, 0x71, 0x54, 0x38,
            0x47, 0x43, 0x6b, 0x55, 0x62, 0x79, 0x77, 0x79, 0x4f, 0x34, 0x39, 0x51, 0x71, 0x6f, 0x2b, 0x62,
            0x55, 0x4f, 0x30, 0x2f, 0x59, 0x30, 0x54, 0x30, 0x38, 0x6e, 0x77, 0x33, 0x33, 0x49, 0x72, 0x44,
            0x30, 0x44, 0x47, 0x71, 0x0a, 0x5a, 0x48, 0x2b, 0x49, 0x73, 0x41, 0x4c, 0x39, 0x34, 0x52, 0x67,
            0x4d, 0x61, 0x78, 0x48, 0x6e, 0x51, 0x6f, 0x48, 0x5a, 0x41, 0x47, 0x2f, 0x44, 0x75, 0x31, 0x36,
            0x6b, 0x2b, 0x6c, 0x41, 0x33, 0x79, 0x63, 0x71, 0x34, 0x51, 0x52, 0x48, 0x73, 0x4e, 0x37, 0x35,
            0x52, 0x32, 0x55, 0x45, 0x43, 0x51, 0x51, 0x44, 0x6d, 0x4f, 0x51, 0x77, 0x34, 0x58, 0x51, 0x4b,
            0x44, 0x41, 0x54, 0x77, 0x48, 0x0a, 0x6b, 0x50, 0x6d, 0x4f, 0x68, 0x33, 0x67, 0x6d, 0x42, 0x65,
            0x39, 0x56, 0x35, 0x49, 0x74, 0x4c, 0x6f, 0x71, 0x2f, 0x59, 0x7a, 0x65, 0x43, 0x6f, 0x41, 0x4f,
            0x5a, 0x68, 0x32, 0x64, 0x4e, 0x47, 0x35, 0x56, 0x52, 0x45, 0x54, 0x6a, 0x61, 0x61, 0x65, 0x45,
            0x50, 0x68, 0x39, 0x66, 0x6f, 0x44, 0x44, 0x33, 0x50, 0x6f, 0x41, 0x72, 0x2f, 0x31, 0x45, 0x57,
            0x74, 0x6a, 0x67, 0x65, 0x55, 0x62, 0x0a, 0x32, 0x51, 0x6b, 0x52, 0x76, 0x70, 0x45, 0x48, 0x41,
            0x6b, 0x45, 0x41, 0x78, 0x47, 0x57, 0x6a, 0x72, 0x4b, 0x53, 0x35, 0x34, 0x56, 0x69, 0x77, 0x56,
            0x5a, 0x45, 0x44, 0x52, 0x31, 0x6b, 0x57, 0x74, 0x42, 0x4f, 0x55, 0x59, 0x59, 0x69, 0x5a, 0x59,
            0x53, 0x58, 0x53, 0x4f, 0x41, 0x73, 0x43, 0x51, 0x2b, 0x59, 0x47, 0x79, 0x36, 0x33, 0x71, 0x54,
            0x59, 0x6c, 0x30, 0x75, 0x6e, 0x34, 0x74, 0x0a, 0x39, 0x46, 0x4f, 0x67, 0x4e, 0x58, 0x41, 0x34,
            0x49, 0x36, 0x35, 0x51, 0x7a, 0x32, 0x2b, 0x57, 0x37, 0x7a, 0x52, 0x79, 0x39, 0x70, 0x47, 0x31,
            0x36, 0x68, 0x4a, 0x62, 0x69, 0x4b, 0x64, 0x69, 0x77, 0x77, 0x4a, 0x41, 0x52, 0x4f, 0x74, 0x42,
            0x43, 0x45, 0x4a, 0x33, 0x74, 0x51, 0x59, 0x53, 0x6c, 0x57, 0x56, 0x46, 0x2f, 0x2b, 0x2f, 0x33,
            0x2f, 0x4e, 0x48, 0x38, 0x41, 0x75, 0x65, 0x47, 0x0a, 0x39, 0x35, 0x35, 0x39, 0x4b, 0x39, 0x62,
            0x39, 0x38, 0x4f, 0x31, 0x4d, 0x43, 0x6e, 0x44, 0x74, 0x2b, 0x61, 0x51, 0x2f, 0x5a, 0x4b, 0x39,
            0x52, 0x55, 0x30, 0x36, 0x37, 0x61, 0x35, 0x4a, 0x36, 0x52, 0x4a, 0x35, 0x45, 0x6f, 0x57, 0x6a,
            0x70, 0x56, 0x42, 0x44, 0x68, 0x36, 0x41, 0x6b, 0x44, 0x73, 0x43, 0x46, 0x35, 0x77, 0x4a, 0x45,
            0x65, 0x30, 0x77, 0x4a, 0x41, 0x4a, 0x38, 0x42, 0x75, 0x0a, 0x31, 0x5a, 0x58, 0x74, 0x37, 0x62,
            0x65, 0x32, 0x30, 0x55, 0x72, 0x74, 0x76, 0x2f, 0x59, 0x2f, 0x53, 0x79, 0x6d, 0x66, 0x59, 0x66,
            0x4a, 0x31, 0x49, 0x52, 0x32, 0x33, 0x70, 0x42, 0x4a, 0x4f, 0x58, 0x68, 0x48, 0x6b, 0x46, 0x44,
            0x76, 0x61, 0x62, 0x65, 0x46, 0x42, 0x6d, 0x44, 0x48, 0x4c, 0x2b, 0x57, 0x55, 0x6e, 0x79, 0x30,
            0x6b, 0x77, 0x67, 0x38, 0x42, 0x58, 0x71, 0x50, 0x54, 0x73, 0x0a, 0x4e, 0x76, 0x53, 0x46, 0x78,
            0x48, 0x53, 0x50, 0x76, 0x70, 0x59, 0x43, 0x6f, 0x76, 0x46, 0x47, 0x59, 0x77, 0x4a, 0x41, 0x4b,
            0x65, 0x33, 0x6a, 0x4d, 0x72, 0x79, 0x74, 0x58, 0x64, 0x53, 0x4f, 0x4b, 0x7a, 0x70, 0x57, 0x4f,
            0x58, 0x4e, 0x69, 0x4e, 0x4b, 0x4b, 0x4a, 0x2f, 0x4e, 0x4a, 0x63, 0x67, 0x72, 0x55, 0x7a, 0x31,
            0x63, 0x39, 0x76, 0x6d, 0x4e, 0x43, 0x6c, 0x32, 0x67, 0x33, 0x45, 0x0a, 0x48, 0x6d, 0x33, 0x45,
            0x4c, 0x30, 0x39, 0x70, 0x41, 0x6a, 0x63, 0x7a, 0x30, 0x63, 0x54, 0x4e, 0x4e, 0x55, 0x6b, 0x63,
            0x6f, 0x5a, 0x44, 0x36, 0x63, 0x4f, 0x31, 0x5a, 0x70, 0x31, 0x68, 0x6e, 0x63, 0x66, 0x52, 0x4b,
            0x4d, 0x70, 0x5a, 0x73, 0x49, 0x41, 0x3d, 0x3d, 0x0a, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x45, 0x4e,
            0x44, 0x20, 0x52, 0x53, 0x41, 0x20, 0x50, 0x52, 0x49, 0x56, 0x41, 0x54, 0x45, 0x20, 0x4b, 0x45,
            0x59, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x0a
        }
    },
    {
        "1024 bit PEM/PKCS1 encrypted", U_SECURITY_CREDENTIAL_CLIENT_KEY_PRIVATE,
        951, U_SECURITY_CREDENTIAL_TEST_PASSPHRASE,
        //lint -e785 Suppress Too few initializers
        {
            0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x42, 0x45, 0x47, 0x49, 0x4e, 0x20, 0x52, 0x53, 0x41, 0x20, 0x50,
            0x52, 0x49, 0x56, 0x41, 0x54, 0x45, 0x20, 0x4b, 0x45, 0x59, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x0a,
            0x50, 0x72, 0x6f, 0x63, 0x2d, 0x54, 0x79, 0x70, 0x65, 0x3a, 0x20, 0x34, 0x2c, 0x45, 0x4e, 0x43,
            0x52, 0x59, 0x50, 0x54, 0x45, 0x44, 0x0a, 0x44, 0x45, 0x4b, 0x2d, 0x49, 0x6e, 0x66, 0x6f, 0x3a,
            0x20, 0x44, 0x45, 0x53, 0x2d, 0x45, 0x44, 0x45, 0x33, 0x2d, 0x43, 0x42, 0x43, 0x2c, 0x36, 0x45,
            0x39, 0x39, 0x45, 0x35, 0x36, 0x38, 0x36, 0x34, 0x42, 0x36, 0x32, 0x44, 0x45, 0x45, 0x0a, 0x0a,
            0x67, 0x56, 0x51, 0x4c, 0x55, 0x37, 0x34, 0x32, 0x72, 0x57, 0x69, 0x6e, 0x70, 0x56, 0x68, 0x33,
            0x64, 0x5a, 0x4e, 0x68, 0x6c, 0x6f, 0x34, 0x43, 0x61, 0x5a, 0x77, 0x4f, 0x6b, 0x4b, 0x44, 0x34,
            0x71, 0x6e, 0x68, 0x68, 0x42, 0x49, 0x6f, 0x4b, 0x36, 0x68, 0x59, 0x6c, 0x44, 0x4c, 0x43, 0x31,
            0x63, 0x7a, 0x33, 0x62, 0x71, 0x7a, 0x73, 0x43, 0x48, 0x34, 0x73, 0x68, 0x6b, 0x70, 0x32, 0x77,
            0x0a, 0x63, 0x5a, 0x63, 0x44, 0x6a, 0x4f, 0x32, 0x65, 0x69, 0x2f, 0x77, 0x69, 0x41, 0x6a, 0x78,
            0x41, 0x62, 0x79, 0x42, 0x78, 0x78, 0x66, 0x64, 0x64, 0x4c, 0x50, 0x4b, 0x50, 0x35, 0x70, 0x74,
            0x72, 0x76, 0x38, 0x65, 0x46, 0x57, 0x79, 0x34, 0x66, 0x2f, 0x75, 0x39, 0x49, 0x6a, 0x63, 0x44,
            0x74, 0x5a, 0x77, 0x71, 0x48, 0x4f, 0x35, 0x2f, 0x56, 0x4b, 0x35, 0x4d, 0x6a, 0x36, 0x75, 0x79,
            0x48, 0x0a, 0x4a, 0x38, 0x58, 0x4b, 0x2f, 0x39, 0x4e, 0x6b, 0x31, 0x31, 0x69, 0x73, 0x68, 0x35,
            0x59, 0x37, 0x59, 0x57, 0x34, 0x72, 0x54, 0x33, 0x47, 0x36, 0x6c, 0x6f, 0x70, 0x50, 0x55, 0x54,
            0x2b, 0x57, 0x55, 0x66, 0x78, 0x35, 0x41, 0x76, 0x77, 0x6b, 0x53, 0x55, 0x46, 0x6d, 0x46, 0x69,
            0x6f, 0x38, 0x33, 0x41, 0x49, 0x78, 0x49, 0x62, 0x67, 0x70, 0x75, 0x74, 0x4a, 0x36, 0x64, 0x71,
            0x51, 0x35, 0x0a, 0x6e, 0x6f, 0x5a, 0x39, 0x57, 0x4e, 0x4a, 0x4b, 0x74, 0x4f, 0x67, 0x59, 0x46,
            0x57, 0x73, 0x43, 0x46, 0x35, 0x73, 0x4b, 0x68, 0x6a, 0x42, 0x2f, 0x73, 0x48, 0x53, 0x64, 0x41,
            0x35, 0x58, 0x54, 0x79, 0x55, 0x5a, 0x69, 0x66, 0x63, 0x43, 0x4e, 0x65, 0x4d, 0x39, 0x48, 0x62,
            0x55, 0x39, 0x73, 0x73, 0x4f, 0x69, 0x6d, 0x34, 0x74, 0x55, 0x4e, 0x73, 0x42, 0x70, 0x37, 0x47,
            0x63, 0x74, 0x33, 0x0a, 0x6a, 0x56, 0x79, 0x55, 0x4d, 0x48, 0x48, 0x35, 0x4c, 0x75, 0x74, 0x53,
            0x68, 0x57, 0x2b, 0x69, 0x35, 0x6f, 0x72, 0x42, 0x62, 0x5a, 0x39, 0x66, 0x7a, 0x51, 0x69, 0x65,
            0x79, 0x43, 0x4e, 0x57, 0x42, 0x4f, 0x38, 0x59, 0x50, 0x39, 0x6e, 0x55, 0x4d, 0x72, 0x62, 0x46,
            0x7a, 0x6c, 0x72, 0x62, 0x73, 0x67, 0x64, 0x32, 0x4e, 0x53, 0x4f, 0x53, 0x4c, 0x4e, 0x4c, 0x59,
            0x2b, 0x59, 0x49, 0x35, 0x0a, 0x52, 0x4d, 0x56, 0x72, 0x63, 0x34, 0x4f, 0x61, 0x6a, 0x74, 0x36,
            0x41, 0x74, 0x7a, 0x54, 0x39, 0x54, 0x57, 0x6a, 0x70, 0x47, 0x36, 0x6e, 0x6c, 0x37, 0x62, 0x44,
            0x5a, 0x42, 0x36, 0x51, 0x43, 0x5a, 0x62, 0x6e, 0x72, 0x74, 0x73, 0x43, 0x56, 0x41, 0x32, 0x35,
            0x49, 0x44, 0x70, 0x72, 0x35, 0x42, 0x34, 0x54, 0x56, 0x64, 0x6c, 0x63, 0x6c, 0x44, 0x31, 0x61,
            0x37, 0x44, 0x61, 0x5a, 0x43, 0x0a, 0x62, 0x43, 0x6b, 0x70, 0x4f, 0x4c, 0x61, 0x74, 0x55, 0x53,
            0x68, 0x53, 0x6a, 0x42, 0x75, 0x32, 0x59, 0x69, 0x49, 0x63, 0x6d, 0x5a, 0x4e, 0x6d, 0x6b, 0x38,
            0x43, 0x4b, 0x62, 0x38, 0x45, 0x6a, 0x4e, 0x6d, 0x30, 0x30, 0x62, 0x51, 0x53, 0x7a, 0x2f, 0x6a,
            0x37, 0x48, 0x64, 0x63, 0x52, 0x57, 0x38, 0x4b, 0x48, 0x41, 0x45, 0x31, 0x7a, 0x56, 0x76, 0x35,
            0x70, 0x57, 0x46, 0x51, 0x53, 0x43, 0x0a, 0x65, 0x6f, 0x42, 0x45, 0x74, 0x51, 0x74, 0x2f, 0x6d,
            0x44, 0x47, 0x55, 0x39, 0x6d, 0x53, 0x59, 0x5a, 0x78, 0x45, 0x38, 0x6f, 0x71, 0x55, 0x62, 0x4d,
            0x31, 0x6a, 0x74, 0x51, 0x56, 0x31, 0x73, 0x6e, 0x67, 0x59, 0x4e, 0x53, 0x44, 0x46, 0x47, 0x74,
            0x2b, 0x58, 0x50, 0x57, 0x7a, 0x6c, 0x50, 0x71, 0x45, 0x34, 0x66, 0x4c, 0x77, 0x48, 0x73, 0x34,
            0x30, 0x4c, 0x6f, 0x57, 0x78, 0x6e, 0x46, 0x0a, 0x6e, 0x47, 0x64, 0x61, 0x75, 0x6c, 0x77, 0x46,
            0x42, 0x49, 0x52, 0x64, 0x58, 0x2f, 0x6c, 0x6e, 0x61, 0x57, 0x55, 0x6e, 0x57, 0x43, 0x41, 0x67,
            0x56, 0x70, 0x73, 0x6b, 0x72, 0x7a, 0x61, 0x59, 0x66, 0x79, 0x6a, 0x58, 0x42, 0x6a, 0x54, 0x46,
            0x4b, 0x31, 0x65, 0x45, 0x78, 0x2b, 0x79, 0x65, 0x68, 0x4d, 0x78, 0x4b, 0x68, 0x75, 0x4b, 0x72,
            0x69, 0x54, 0x6d, 0x52, 0x31, 0x54, 0x58, 0x45, 0x0a, 0x51, 0x57, 0x6d, 0x71, 0x51, 0x4c, 0x4c,
            0x77, 0x51, 0x51, 0x68, 0x77, 0x37, 0x72, 0x2f, 0x47, 0x50, 0x67, 0x31, 0x76, 0x58, 0x33, 0x49,
            0x54, 0x78, 0x30, 0x6a, 0x38, 0x42, 0x64, 0x69, 0x67, 0x71, 0x32, 0x74, 0x51, 0x61, 0x75, 0x73,
            0x34, 0x66, 0x32, 0x63, 0x38, 0x2b, 0x57, 0x32, 0x4e, 0x45, 0x63, 0x6a, 0x4b, 0x4c, 0x30, 0x39,
            0x41, 0x2b, 0x7a, 0x2f, 0x6a, 0x66, 0x42, 0x4a, 0x6e, 0x0a, 0x79, 0x59, 0x79, 0x6e, 0x52, 0x6d,
            0x47, 0x33, 0x4a, 0x52, 0x2b, 0x53, 0x4f, 0x70, 0x57, 0x6a, 0x52, 0x48, 0x2f, 0x41, 0x30, 0x79,
            0x51, 0x45, 0x4d, 0x46, 0x59, 0x78, 0x58, 0x2f, 0x4c, 0x4c, 0x72, 0x71, 0x71, 0x67, 0x41, 0x41,
            0x77, 0x45, 0x73, 0x74, 0x2f, 0x53, 0x68, 0x51, 0x64, 0x5a, 0x2f, 0x56, 0x2f, 0x2b, 0x32, 0x6e,
            0x4b, 0x70, 0x61, 0x6e, 0x67, 0x2f, 0x54, 0x79, 0x53, 0x56, 0x0a, 0x30, 0x32, 0x77, 0x74, 0x70,
            0x72, 0x65, 0x7a, 0x77, 0x74, 0x53, 0x49, 0x62, 0x4a, 0x51, 0x71, 0x2f, 0x57, 0x75, 0x4e, 0x39,
            0x6e, 0x65, 0x33, 0x52, 0x59, 0x75, 0x6b, 0x49, 0x65, 0x4f, 0x41, 0x53, 0x50, 0x33, 0x6c, 0x46,
            0x45, 0x36, 0x53, 0x44, 0x54, 0x35, 0x6b, 0x43, 0x74, 0x61, 0x35, 0x56, 0x51, 0x79, 0x4f, 0x4a,
            0x6a, 0x33, 0x6d, 0x61, 0x34, 0x70, 0x65, 0x4d, 0x30, 0x32, 0x43, 0x0a, 0x6f, 0x46, 0x78, 0x45,
            0x79, 0x37, 0x31, 0x37, 0x6f, 0x6a, 0x77, 0x61, 0x35, 0x39, 0x33, 0x45, 0x63, 0x33, 0x51, 0x56,
            0x34, 0x76, 0x33, 0x34, 0x58, 0x71, 0x69, 0x52, 0x72, 0x6d, 0x74, 0x79, 0x41, 0x56, 0x5a, 0x39,
            0x72, 0x4b, 0x7a, 0x68, 0x39, 0x76, 0x4d, 0x3d, 0x0a, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x45, 0x4e,
            0x44, 0x20, 0x52, 0x53, 0x41, 0x20, 0x50, 0x52, 0x49, 0x56, 0x41, 0x54, 0x45, 0x20, 0x4b, 0x45,
            0x59, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x0a
        }
    },
    {
        "1024 bit PEM/PKCS8", U_SECURITY_CREDENTIAL_CLIENT_KEY_PRIVATE,
        912, NULL,
        //lint -e785 Suppress Too few initializers
        {
            0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x42, 0x45, 0x47, 0x49, 0x4e, 0x20, 0x50, 0x52, 0x49, 0x56, 0x41,
            0x54, 0x45, 0x20, 0x4b, 0x45, 0x59, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x0a, 0x4d, 0x49, 0x49, 0x43,
            0x64, 0x51, 0x49, 0x42, 0x41, 0x44, 0x41, 0x4e, 0x42, 0x67, 0x6b, 0x71, 0x68, 0x6b, 0x69, 0x47,
            0x39, 0x77, 0x30, 0x42, 0x41, 0x51, 0x45, 0x46, 0x41, 0x41, 0x53, 0x43, 0x41, 0x6c, 0x38, 0x77,
            0x67, 0x67, 0x4a, 0x62, 0x41, 0x67, 0x45, 0x41, 0x41, 0x6f, 0x47, 0x42, 0x41, 0x4c, 0x43, 0x66,
            0x46, 0x51, 0x36, 0x4f, 0x73, 0x6d, 0x30, 0x58, 0x6a, 0x73, 0x73, 0x44, 0x0a, 0x70, 0x68, 0x2b,
            0x39, 0x4e, 0x30, 0x51, 0x4b, 0x7a, 0x76, 0x77, 0x51, 0x5a, 0x45, 0x56, 0x6a, 0x54, 0x69, 0x64,
            0x6b, 0x68, 0x31, 0x2b, 0x48, 0x70, 0x42, 0x53, 0x45, 0x45, 0x62, 0x33, 0x61, 0x36, 0x4b, 0x7a,
            0x66, 0x4c, 0x48, 0x4c, 0x58, 0x57, 0x50, 0x7a, 0x46, 0x46, 0x56, 0x56, 0x77, 0x6c, 0x6a, 0x75,
            0x51, 0x73, 0x41, 0x77, 0x72, 0x4c, 0x42, 0x59, 0x76, 0x52, 0x54, 0x59, 0x69, 0x0a, 0x31, 0x76,
            0x6a, 0x45, 0x44, 0x44, 0x32, 0x45, 0x33, 0x75, 0x58, 0x50, 0x79, 0x47, 0x63, 0x73, 0x73, 0x36,
            0x41, 0x6b, 0x68, 0x39, 0x6d, 0x54, 0x52, 0x36, 0x55, 0x47, 0x53, 0x36, 0x48, 0x77, 0x32, 0x48,
            0x46, 0x6c, 0x48, 0x33, 0x50, 0x57, 0x44, 0x39, 0x34, 0x4c, 0x52, 0x6a, 0x69, 0x52, 0x44, 0x77,
            0x4e, 0x6e, 0x36, 0x43, 0x34, 0x68, 0x70, 0x72, 0x61, 0x6a, 0x30, 0x70, 0x5a, 0x52, 0x0a, 0x62,
            0x49, 0x42, 0x45, 0x37, 0x75, 0x57, 0x46, 0x66, 0x4e, 0x42, 0x39, 0x71, 0x38, 0x46, 0x6a, 0x52,
            0x38, 0x76, 0x35, 0x70, 0x6f, 0x7a, 0x4c, 0x50, 0x69, 0x5a, 0x56, 0x41, 0x67, 0x4d, 0x42, 0x41,
            0x41, 0x45, 0x43, 0x67, 0x59, 0x41, 0x44, 0x77, 0x46, 0x31, 0x43, 0x49, 0x65, 0x58, 0x49, 0x53,
            0x32, 0x66, 0x4e, 0x43, 0x42, 0x62, 0x50, 0x4e, 0x63, 0x57, 0x6d, 0x76, 0x46, 0x51, 0x4a, 0x0a,
            0x4a, 0x6f, 0x2f, 0x36, 0x71, 0x56, 0x46, 0x50, 0x70, 0x4f, 0x4c, 0x37, 0x4e, 0x48, 0x5a, 0x58,
            0x75, 0x54, 0x5a, 0x6e, 0x51, 0x65, 0x38, 0x50, 0x50, 0x4f, 0x35, 0x45, 0x77, 0x69, 0x34, 0x78,
            0x73, 0x44, 0x33, 0x7a, 0x77, 0x71, 0x42, 0x75, 0x51, 0x33, 0x52, 0x35, 0x7a, 0x62, 0x44, 0x79,
            0x42, 0x36, 0x38, 0x41, 0x70, 0x31, 0x7a, 0x4a, 0x69, 0x2b, 0x42, 0x32, 0x70, 0x50, 0x77, 0x59,
            0x0a, 0x4b, 0x52, 0x52, 0x76, 0x4c, 0x44, 0x49, 0x37, 0x6a, 0x31, 0x43, 0x71, 0x6a, 0x35, 0x74,
            0x51, 0x37, 0x54, 0x39, 0x6a, 0x52, 0x50, 0x54, 0x79, 0x66, 0x44, 0x66, 0x63, 0x69, 0x73, 0x50,
            0x51, 0x4d, 0x61, 0x70, 0x6b, 0x66, 0x34, 0x69, 0x77, 0x41, 0x76, 0x33, 0x68, 0x47, 0x41, 0x78,
            0x72, 0x45, 0x65, 0x64, 0x43, 0x67, 0x64, 0x6b, 0x41, 0x62, 0x38, 0x4f, 0x37, 0x58, 0x71, 0x54,
            0x36, 0x0a, 0x55, 0x44, 0x66, 0x4a, 0x79, 0x72, 0x68, 0x42, 0x45, 0x65, 0x77, 0x33, 0x76, 0x6c,
            0x48, 0x5a, 0x51, 0x51, 0x4a, 0x42, 0x41, 0x4f, 0x59, 0x35, 0x44, 0x44, 0x68, 0x64, 0x41, 0x6f,
            0x4d, 0x42, 0x50, 0x41, 0x65, 0x51, 0x2b, 0x59, 0x36, 0x48, 0x65, 0x43, 0x59, 0x46, 0x37, 0x31,
            0x58, 0x6b, 0x69, 0x30, 0x75, 0x69, 0x72, 0x39, 0x6a, 0x4e, 0x34, 0x4b, 0x67, 0x41, 0x35, 0x6d,
            0x48, 0x5a, 0x0a, 0x30, 0x30, 0x62, 0x6c, 0x56, 0x45, 0x52, 0x4f, 0x4e, 0x70, 0x70, 0x34, 0x51,
            0x2b, 0x48, 0x31, 0x2b, 0x67, 0x4d, 0x50, 0x63, 0x2b, 0x67, 0x43, 0x76, 0x2f, 0x55, 0x52, 0x61,
            0x32, 0x4f, 0x42, 0x35, 0x52, 0x76, 0x5a, 0x43, 0x52, 0x47, 0x2b, 0x6b, 0x51, 0x63, 0x43, 0x51,
            0x51, 0x44, 0x45, 0x5a, 0x61, 0x4f, 0x73, 0x70, 0x4c, 0x6e, 0x68, 0x57, 0x4c, 0x42, 0x56, 0x6b,
            0x51, 0x4e, 0x48, 0x0a, 0x57, 0x52, 0x61, 0x30, 0x45, 0x35, 0x52, 0x68, 0x69, 0x4a, 0x6c, 0x68,
            0x4a, 0x64, 0x49, 0x34, 0x43, 0x77, 0x4a, 0x44, 0x35, 0x67, 0x62, 0x4c, 0x72, 0x65, 0x70, 0x4e,
            0x69, 0x58, 0x53, 0x36, 0x66, 0x69, 0x33, 0x30, 0x55, 0x36, 0x41, 0x31, 0x63, 0x44, 0x67, 0x6a,
            0x72, 0x6c, 0x44, 0x50, 0x62, 0x35, 0x62, 0x76, 0x4e, 0x48, 0x4c, 0x32, 0x6b, 0x62, 0x58, 0x71,
            0x45, 0x6c, 0x75, 0x49, 0x0a, 0x70, 0x32, 0x4c, 0x44, 0x41, 0x6b, 0x42, 0x45, 0x36, 0x30, 0x45,
            0x49, 0x51, 0x6e, 0x65, 0x31, 0x42, 0x68, 0x4b, 0x56, 0x5a, 0x55, 0x58, 0x2f, 0x37, 0x2f, 0x66,
            0x38, 0x30, 0x66, 0x77, 0x43, 0x35, 0x34, 0x62, 0x33, 0x6e, 0x6e, 0x30, 0x72, 0x31, 0x76, 0x33,
            0x77, 0x37, 0x55, 0x77, 0x4b, 0x63, 0x4f, 0x33, 0x35, 0x70, 0x44, 0x39, 0x6b, 0x72, 0x31, 0x46,
            0x54, 0x54, 0x72, 0x74, 0x72, 0x0a, 0x6b, 0x6e, 0x70, 0x45, 0x6e, 0x6b, 0x53, 0x68, 0x61, 0x4f,
            0x6c, 0x55, 0x45, 0x4f, 0x48, 0x6f, 0x43, 0x51, 0x4f, 0x77, 0x49, 0x58, 0x6e, 0x41, 0x6b, 0x52,
            0x37, 0x54, 0x41, 0x6b, 0x41, 0x6e, 0x77, 0x47, 0x37, 0x56, 0x6c, 0x65, 0x33, 0x74, 0x74, 0x37,
            0x62, 0x52, 0x53, 0x75, 0x32, 0x2f, 0x39, 0x6a, 0x39, 0x4c, 0x4b, 0x5a, 0x39, 0x68, 0x38, 0x6e,
            0x55, 0x68, 0x48, 0x62, 0x65, 0x6b, 0x0a, 0x45, 0x6b, 0x35, 0x65, 0x45, 0x65, 0x51, 0x55, 0x4f,
            0x39, 0x70, 0x74, 0x34, 0x55, 0x47, 0x59, 0x4d, 0x63, 0x76, 0x35, 0x5a, 0x53, 0x66, 0x4c, 0x53,
            0x54, 0x43, 0x44, 0x77, 0x46, 0x65, 0x6f, 0x39, 0x4f, 0x77, 0x32, 0x39, 0x49, 0x58, 0x45, 0x64,
            0x49, 0x2b, 0x2b, 0x6c, 0x67, 0x4b, 0x69, 0x38, 0x55, 0x5a, 0x6a, 0x41, 0x6b, 0x41, 0x70, 0x37,
            0x65, 0x4d, 0x79, 0x76, 0x4b, 0x31, 0x64, 0x0a, 0x31, 0x49, 0x34, 0x72, 0x4f, 0x6c, 0x59, 0x35,
            0x63, 0x32, 0x49, 0x30, 0x6f, 0x6f, 0x6e, 0x38, 0x30, 0x6c, 0x79, 0x43, 0x74, 0x54, 0x50, 0x56,
            0x7a, 0x32, 0x2b, 0x59, 0x30, 0x4b, 0x58, 0x61, 0x44, 0x63, 0x51, 0x65, 0x62, 0x63, 0x51, 0x76,
            0x54, 0x32, 0x6b, 0x43, 0x4e, 0x7a, 0x50, 0x52, 0x78, 0x4d, 0x30, 0x31, 0x53, 0x52, 0x79, 0x68,
            0x6b, 0x50, 0x70, 0x77, 0x37, 0x56, 0x6d, 0x6e, 0x0a, 0x57, 0x47, 0x64, 0x78, 0x39, 0x45, 0x6f,
            0x79, 0x6c, 0x6d, 0x77, 0x67, 0x0a, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x45, 0x4e, 0x44, 0x20, 0x50,
            0x52, 0x49, 0x56, 0x41, 0x54, 0x45, 0x20, 0x4b, 0x45, 0x59, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x0a
        }
    },
    {
        "1024 bit PEM/PKCS8 encrypted", U_SECURITY_CREDENTIAL_CLIENT_KEY_PRIVATE,
        993, U_SECURITY_CREDENTIAL_TEST_PASSPHRASE,
        //lint -e785 Suppress Too few initializers
        {
            0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x42, 0x45, 0x47, 0x49, 0x4e, 0x20, 0x45, 0x4e, 0x43, 0x52, 0x59,
            0x50, 0x54, 0x45, 0x44, 0x20, 0x50, 0x52, 0x49, 0x56, 0x41, 0x54, 0x45, 0x20, 0x4b, 0x45, 0x59,
            0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x0a, 0x4d, 0x49, 0x49, 0x43, 0x6f, 0x54, 0x41, 0x62, 0x42, 0x67,
            0x6b, 0x71, 0x68, 0x6b, 0x69, 0x47, 0x39, 0x77, 0x30, 0x42, 0x42, 0x51, 0x4d, 0x77, 0x44, 0x67,
            0x51, 0x49, 0x4d, 0x79, 0x2b, 0x2f, 0x32, 0x36, 0x6d, 0x59, 0x31, 0x4d, 0x67, 0x43, 0x41, 0x67,
            0x67, 0x41, 0x42, 0x49, 0x49, 0x43, 0x67, 0x4a, 0x7a, 0x50, 0x44, 0x6e, 0x2f, 0x76, 0x52, 0x68,
            0x45, 0x61, 0x68, 0x52, 0x6e, 0x4c, 0x0a, 0x56, 0x39, 0x49, 0x6e, 0x32, 0x39, 0x6f, 0x75, 0x31,
            0x54, 0x34, 0x76, 0x36, 0x77, 0x50, 0x68, 0x6c, 0x53, 0x79, 0x4b, 0x4d, 0x33, 0x39, 0x6e, 0x45,
            0x76, 0x4c, 0x2b, 0x72, 0x51, 0x76, 0x4c, 0x6d, 0x31, 0x52, 0x77, 0x59, 0x6e, 0x4a, 0x4c, 0x52,
            0x69, 0x42, 0x63, 0x48, 0x30, 0x48, 0x75, 0x6f, 0x63, 0x49, 0x2f, 0x51, 0x58, 0x33, 0x76, 0x52,
            0x43, 0x41, 0x42, 0x76, 0x4d, 0x77, 0x6f, 0x0a, 0x30, 0x4e, 0x69, 0x4a, 0x35, 0x54, 0x45, 0x67,
            0x47, 0x56, 0x34, 0x71, 0x70, 0x6d, 0x50, 0x76, 0x67, 0x4d, 0x6f, 0x62, 0x7a, 0x4b, 0x39, 0x6b,
            0x63, 0x33, 0x78, 0x5a, 0x65, 0x36, 0x51, 0x32, 0x36, 0x35, 0x4c, 0x6d, 0x32, 0x62, 0x78, 0x49,
            0x76, 0x53, 0x73, 0x4d, 0x6a, 0x32, 0x4d, 0x30, 0x64, 0x45, 0x31, 0x32, 0x59, 0x51, 0x53, 0x61,
            0x49, 0x47, 0x6b, 0x57, 0x66, 0x73, 0x66, 0x2f, 0x0a, 0x6b, 0x2f, 0x5a, 0x4c, 0x68, 0x49, 0x65,
            0x46, 0x67, 0x2b, 0x57, 0x37, 0x69, 0x34, 0x4e, 0x50, 0x6a, 0x38, 0x65, 0x42, 0x37, 0x66, 0x75,
            0x79, 0x6d, 0x70, 0x39, 0x41, 0x73, 0x5a, 0x34, 0x57, 0x62, 0x56, 0x4c, 0x2b, 0x37, 0x47, 0x4b,
            0x71, 0x77, 0x32, 0x66, 0x59, 0x63, 0x58, 0x4b, 0x4e, 0x46, 0x46, 0x42, 0x56, 0x6c, 0x53, 0x4a,
            0x7a, 0x61, 0x41, 0x37, 0x42, 0x2f, 0x52, 0x51, 0x76, 0x0a, 0x32, 0x42, 0x41, 0x4a, 0x64, 0x69,
            0x6a, 0x77, 0x4f, 0x76, 0x76, 0x6b, 0x39, 0x59, 0x78, 0x6e, 0x5a, 0x72, 0x50, 0x41, 0x37, 0x73,
            0x74, 0x47, 0x44, 0x30, 0x33, 0x67, 0x44, 0x4c, 0x68, 0x61, 0x6e, 0x54, 0x63, 0x5a, 0x37, 0x39,
            0x46, 0x6e, 0x34, 0x64, 0x35, 0x46, 0x57, 0x57, 0x69, 0x42, 0x4a, 0x51, 0x55, 0x64, 0x34, 0x32,
            0x55, 0x63, 0x4c, 0x4f, 0x68, 0x53, 0x57, 0x62, 0x55, 0x42, 0x0a, 0x56, 0x4b, 0x75, 0x58, 0x75,
            0x4a, 0x6d, 0x53, 0x4f, 0x73, 0x5a, 0x45, 0x68, 0x73, 0x38, 0x36, 0x76, 0x61, 0x2b, 0x7a, 0x58,
            0x65, 0x49, 0x61, 0x5a, 0x51, 0x6e, 0x62, 0x65, 0x30, 0x7a, 0x46, 0x62, 0x71, 0x75, 0x76, 0x45,
            0x43, 0x65, 0x78, 0x72, 0x4e, 0x49, 0x50, 0x46, 0x2b, 0x75, 0x6d, 0x62, 0x75, 0x64, 0x62, 0x4e,
            0x65, 0x77, 0x4d, 0x5a, 0x33, 0x72, 0x6b, 0x6e, 0x61, 0x52, 0x42, 0x0a, 0x63, 0x69, 0x73, 0x4b,
            0x6f, 0x4d, 0x2f, 0x4b, 0x71, 0x32, 0x58, 0x36, 0x55, 0x52, 0x6e, 0x74, 0x77, 0x5a, 0x30, 0x2b,
            0x79, 0x38, 0x56, 0x61, 0x73, 0x45, 0x55, 0x59, 0x4d, 0x79, 0x4b, 0x45, 0x4f, 0x6d, 0x53, 0x74,
            0x4d, 0x46, 0x59, 0x7a, 0x47, 0x42, 0x50, 0x72, 0x52, 0x77, 0x2b, 0x6b, 0x38, 0x5a, 0x65, 0x57,
            0x32, 0x50, 0x77, 0x6a, 0x48, 0x75, 0x2b, 0x61, 0x65, 0x31, 0x75, 0x6c, 0x0a, 0x30, 0x6c, 0x6a,
            0x68, 0x53, 0x30, 0x62, 0x47, 0x78, 0x4a, 0x31, 0x76, 0x6d, 0x67, 0x52, 0x41, 0x70, 0x2f, 0x64,
            0x66, 0x57, 0x47, 0x73, 0x37, 0x73, 0x72, 0x51, 0x76, 0x47, 0x73, 0x74, 0x4e, 0x4f, 0x4b, 0x72,
            0x32, 0x43, 0x70, 0x51, 0x71, 0x51, 0x46, 0x58, 0x44, 0x37, 0x4e, 0x4a, 0x37, 0x37, 0x4c, 0x37,
            0x66, 0x6a, 0x54, 0x4f, 0x32, 0x74, 0x48, 0x79, 0x7a, 0x70, 0x46, 0x2b, 0x4f, 0x0a, 0x77, 0x37,
            0x38, 0x46, 0x50, 0x61, 0x31, 0x36, 0x42, 0x64, 0x35, 0x4e, 0x4c, 0x72, 0x58, 0x48, 0x69, 0x4b,
            0x41, 0x63, 0x67, 0x6b, 0x51, 0x6c, 0x45, 0x5a, 0x6a, 0x71, 0x30, 0x46, 0x65, 0x4e, 0x54, 0x4a,
            0x71, 0x53, 0x47, 0x4b, 0x6f, 0x2b, 0x64, 0x43, 0x71, 0x45, 0x46, 0x70, 0x62, 0x47, 0x70, 0x6f,
            0x78, 0x30, 0x4f, 0x41, 0x31, 0x34, 0x61, 0x48, 0x4f, 0x32, 0x4f, 0x2f, 0x4a, 0x73, 0x0a, 0x61,
            0x77, 0x78, 0x67, 0x57, 0x6d, 0x63, 0x64, 0x34, 0x57, 0x38, 0x44, 0x35, 0x51, 0x56, 0x65, 0x77,
            0x57, 0x31, 0x6c, 0x74, 0x67, 0x76, 0x41, 0x54, 0x4f, 0x36, 0x79, 0x45, 0x31, 0x6c, 0x4b, 0x74,
            0x78, 0x68, 0x50, 0x34, 0x79, 0x65, 0x6a, 0x74, 0x50, 0x65, 0x59, 0x46, 0x52, 0x42, 0x69, 0x6d,
            0x63, 0x69, 0x63, 0x35, 0x77, 0x41, 0x30, 0x43, 0x68, 0x59, 0x68, 0x77, 0x73, 0x35, 0x62, 0x0a,
            0x41, 0x37, 0x77, 0x4e, 0x45, 0x30, 0x79, 0x31, 0x38, 0x48, 0x6a, 0x4a, 0x6d, 0x70, 0x63, 0x69,
            0x55, 0x43, 0x71, 0x35, 0x32, 0x62, 0x4e, 0x47, 0x50, 0x6a, 0x54, 0x34, 0x41, 0x30, 0x41, 0x7a,
            0x37, 0x69, 0x71, 0x42, 0x32, 0x38, 0x4a, 0x4f, 0x58, 0x65, 0x41, 0x53, 0x73, 0x66, 0x6f, 0x33,
            0x45, 0x73, 0x4f, 0x41, 0x4a, 0x39, 0x59, 0x45, 0x65, 0x37, 0x5a, 0x6c, 0x30, 0x4d, 0x49, 0x2b,
            0x0a, 0x63, 0x4c, 0x64, 0x67, 0x58, 0x37, 0x42, 0x67, 0x78, 0x44, 0x2b, 0x61, 0x5a, 0x32, 0x76,
            0x30, 0x77, 0x4f, 0x6a, 0x50, 0x52, 0x4b, 0x71, 0x6f, 0x4c, 0x70, 0x33, 0x72, 0x64, 0x52, 0x66,
            0x47, 0x44, 0x61, 0x54, 0x45, 0x70, 0x45, 0x71, 0x76, 0x75, 0x6f, 0x36, 0x46, 0x38, 0x41, 0x70,
            0x32, 0x61, 0x4e, 0x52, 0x58, 0x4a, 0x35, 0x46, 0x6f, 0x44, 0x44, 0x51, 0x53, 0x7a, 0x69, 0x75,
            0x51, 0x0a, 0x35, 0x72, 0x4a, 0x7a, 0x50, 0x35, 0x7a, 0x68, 0x64, 0x4e, 0x59, 0x71, 0x71, 0x4f,
            0x41, 0x52, 0x77, 0x35, 0x71, 0x59, 0x32, 0x34, 0x2b, 0x42, 0x62, 0x6a, 0x48, 0x65, 0x66, 0x49,
            0x57, 0x74, 0x31, 0x33, 0x54, 0x31, 0x62, 0x58, 0x45, 0x49, 0x4a, 0x59, 0x4d, 0x46, 0x46, 0x6d,
            0x70, 0x39, 0x57, 0x35, 0x49, 0x73, 0x68, 0x63, 0x79, 0x35, 0x30, 0x61, 0x37, 0x4c, 0x64, 0x58,
            0x6a, 0x43, 0x0a, 0x65, 0x43, 0x70, 0x57, 0x71, 0x32, 0x69, 0x2b, 0x4e, 0x74, 0x6f, 0x52, 0x4e,
            0x48, 0x71, 0x32, 0x4f, 0x38, 0x64, 0x58, 0x6f, 0x42, 0x37, 0x47, 0x77, 0x53, 0x36, 0x2b, 0x6f,
            0x61, 0x70, 0x58, 0x6d, 0x55, 0x56, 0x56, 0x41, 0x5a, 0x7a, 0x62, 0x4b, 0x6c, 0x54, 0x55, 0x4d,
            0x6a, 0x62, 0x2f, 0x54, 0x2f, 0x73, 0x46, 0x51, 0x4c, 0x65, 0x47, 0x4e, 0x44, 0x6c, 0x36, 0x39,
            0x62, 0x39, 0x35, 0x0a, 0x30, 0x7a, 0x44, 0x6d, 0x46, 0x37, 0x41, 0x3d, 0x0a, 0x2d, 0x2d, 0x2d,
            0x2d, 0x2d, 0x45, 0x4e, 0x44, 0x20, 0x45, 0x4e, 0x43, 0x52, 0x59, 0x50, 0x54, 0x45, 0x44, 0x20,
            0x50, 0x52, 0x49, 0x56, 0x41, 0x54, 0x45, 0x20, 0x4b, 0x45, 0x59, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d,
            0x0a
        }
    },
    {
        "1024 bit DER/PKCS1", U_SECURITY_CREDENTIAL_CLIENT_KEY_PRIVATE,
        607, NULL,
        //lint -e785 Suppress Too few initializers
        {
            0x30, 0x82, 0x02, 0x5b, 0x02, 0x01, 0x20, 0x02, 0x81, 0x81, 0x20, 0xb0, 0x9f, 0x15, 0x0e, 0x8e,
            0xb2, 0x6d, 0x17, 0x8e, 0xcb, 0x03, 0xa6, 0x1f, 0xbd, 0x37, 0x44, 0x0a, 0xce, 0xfc, 0x10, 0x64,
            0x45, 0x63, 0x4e, 0x27, 0x64, 0x87, 0x5f, 0x87, 0xa4, 0x14, 0x84, 0x11, 0xbd, 0xda, 0xe8, 0xac,
            0xdf, 0x2c, 0x72, 0xd7, 0x58, 0xfc, 0xc5, 0x15, 0x55, 0x70, 0x96, 0x3b, 0x90, 0xb0, 0x0c, 0x2b,
            0x2c, 0x16, 0x2f, 0x45, 0x36, 0x22, 0xd6, 0xf8, 0xc4, 0x0c, 0x3d, 0x84, 0xde, 0xe5, 0xcf, 0xc8,
            0x67, 0x2c, 0xb3, 0xa0, 0x24, 0x87, 0xd9, 0x93, 0x47, 0xa5, 0x06, 0x4b, 0xa1, 0xf0, 0xd8, 0x71,
            0x65, 0x1f, 0x73, 0xd6, 0x0f, 0xde, 0x0b, 0x46, 0x38, 0x91, 0x0f, 0x03, 0x67, 0xe8, 0x2e, 0x21,
            0xa6, 0xb6, 0xa3, 0xd2, 0x96, 0x51, 0x6c, 0x80, 0x44, 0xee, 0xe5, 0x85, 0x7c, 0xd0, 0x7d, 0xab,
            0xc1, 0x63, 0x47, 0xcb, 0xf9, 0xa6, 0x8c, 0xcb, 0x3e, 0x26, 0x55, 0x02, 0x03, 0x01, 0x20, 0x01,
            0x02, 0x81, 0x80, 0x03, 0xc0, 0x5d, 0x42, 0x21, 0xe5, 0xc8, 0x4b, 0x67, 0xcd, 0x08, 0x16, 0xcf,
            0x35, 0xc5, 0xa6, 0xbc, 0x54, 0x09, 0x26, 0x8f, 0xfa, 0xa9, 0x51, 0x4f, 0xa4, 0xe2, 0xfb, 0x34,
            0x76, 0x57, 0xb9, 0x36, 0x67, 0x41, 0xef, 0x0f, 0x3c, 0xee, 0x44, 0xc2, 0x2e, 0x31, 0xb0, 0x3d,
            0xf3, 0xc2, 0xa0, 0x6e, 0x43, 0x74, 0x79, 0xcd, 0xb0, 0xf2, 0x07, 0xaf, 0x20, 0xa7, 0x5c, 0xc9,
            0x8b, 0xe0, 0x76, 0xa4, 0xfc, 0x18, 0x29, 0x14, 0x6f, 0x2c, 0x32, 0x3b, 0x8f, 0x50, 0xaa, 0x8f,
            0x9b, 0x50, 0xed, 0x3f, 0x63, 0x44, 0xf4, 0xf2, 0x7c, 0x37, 0xdc, 0x8a, 0xc3, 0xd0, 0x31, 0xaa,
            0x64, 0x7f, 0x88, 0xb0, 0x02, 0xfd, 0xe1, 0x18, 0x0c, 0x6b, 0x11, 0xe7, 0x42, 0x81, 0xd9, 0x20,
            0x6f, 0xc3, 0xbb, 0x5e, 0xa4, 0xfa, 0x50, 0x37, 0xc9, 0xca, 0xb8, 0x41, 0x11, 0xec, 0x37, 0xbe,
            0x51, 0xd9, 0x41, 0x02, 0x41, 0x20, 0xe6, 0x39, 0x0c, 0x38, 0x5d, 0x02, 0x83, 0x01, 0x3c, 0x07,
            0x90, 0xf9, 0x8e, 0x87, 0x78, 0x26, 0x05, 0xef, 0x55, 0xe4, 0x8b, 0x4b, 0xa2, 0xaf, 0xd8, 0xcd,
            0xe0, 0xa8, 0x20, 0xe6, 0x61, 0xd9, 0xd3, 0x46, 0xe5, 0x54, 0x44, 0x4e, 0x36, 0x9a, 0x78, 0x43,
            0xe1, 0xf5, 0xfa, 0x03, 0x0f, 0x73, 0xe8, 0x02, 0xbf, 0xf5, 0x11, 0x6b, 0x63, 0x81, 0xe5, 0x1b,
            0xd9, 0x09, 0x11, 0xbe, 0x91, 0x07, 0x02, 0x41, 0x20, 0xc4, 0x65, 0xa3, 0xac, 0xa4, 0xb9, 0xe1,
            0x58, 0xb0, 0x55, 0x91, 0x03, 0x47, 0x59, 0x16, 0xb4, 0x13, 0x94, 0x61, 0x88, 0x99, 0x61, 0x25,
            0xd2, 0x38, 0x0b, 0x02, 0x43, 0xe6, 0x06, 0xcb, 0xad, 0xea, 0x4d, 0x89, 0x74, 0xba, 0x7e, 0x2d,
            0xf4, 0x53, 0xa0, 0x35, 0x70, 0x38, 0x23, 0xae, 0x50, 0xcf, 0x6f, 0x96, 0xef, 0x34, 0x72, 0xf6,
            0x91, 0xb5, 0xea, 0x12, 0x5b, 0x88, 0xa7, 0x62, 0xc3, 0x02, 0x40, 0x44, 0xeb, 0x41, 0x08, 0x42,
            0x77, 0xb5, 0x06, 0x12, 0x95, 0x65, 0x45, 0xff, 0xef, 0xf7, 0xfc, 0xd1, 0xfc, 0x02, 0xe7, 0x86,
            0xf7, 0x9e, 0x7d, 0x2b, 0xd6, 0xfd, 0xf0, 0xed, 0x4c, 0x0a, 0x70, 0xed, 0xf9, 0xa4, 0x3f, 0x64,
            0xaf, 0x51, 0x53, 0x4e, 0xbb, 0x6b, 0x92, 0x7a, 0x44, 0x9e, 0x44, 0xa1, 0x68, 0xe9, 0x54, 0x10,
            0xe1, 0xe8, 0x09, 0x03, 0xb0, 0x21, 0x79, 0xc0, 0x91, 0x1e, 0xd3, 0x02, 0x40, 0x27, 0xc0, 0x6e,
            0xd5, 0x95, 0xed, 0xed, 0xb7, 0xb6, 0xd1, 0x4a, 0xed, 0xbf, 0xf6, 0x3f, 0x4b, 0x29, 0x9f, 0x61,
            0xf2, 0x75, 0x21, 0x1d, 0xb7, 0xa4, 0x12, 0x4e, 0x5e, 0x11, 0xe4, 0x14, 0x3b, 0xda, 0x6d, 0xe1,
            0x41, 0x98, 0x31, 0xcb, 0xf9, 0x65, 0x27, 0xcb, 0x49, 0x30, 0x83, 0xc0, 0x57, 0xa8, 0xf4, 0xec,
            0x36, 0xf4, 0x85, 0xc4, 0x74, 0x8f, 0xbe, 0x96, 0x02, 0xa2, 0xf1, 0x46, 0x63, 0x02, 0x40, 0x29,
            0xed, 0xe3, 0x32, 0xbc, 0xad, 0x5d, 0xd4, 0x8e, 0x2b, 0x3a, 0x56, 0x39, 0x73, 0x62, 0x34, 0xa2,
            0x89, 0xfc, 0xd2, 0x5c, 0x82, 0xb5, 0x33, 0xd5, 0xcf, 0x6f, 0x98, 0xd0, 0xa5, 0xda, 0x0d, 0xc4,
            0x1e, 0x6d, 0xc4, 0x2f, 0x4f, 0x69, 0x02, 0x37, 0x33, 0xd1, 0xc4, 0xcd, 0x35, 0x49, 0x1c, 0xa1,
            0x90, 0xfa, 0x70, 0xed, 0x59, 0xa7, 0x58, 0x67, 0x71, 0xf4, 0x4a, 0x32, 0x96, 0x6c, 0x20
        }
    },
    {
        "1024 bit DER/PKCS8", U_SECURITY_CREDENTIAL_CLIENT_KEY_PRIVATE,
        633, NULL,
        //lint -e785 Suppress Too few initializers
        {
            0x30, 0x82, 0x02, 0x75, 0x02, 0x01, 0x20, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7,
            0x0d, 0x01, 0x01, 0x01, 0x05, 0x20, 0x04, 0x82, 0x02, 0x5f, 0x30, 0x82, 0x02, 0x5b, 0x02, 0x01,
            0x20, 0x02, 0x81, 0x81, 0x20, 0xb0, 0x9f, 0x15, 0x0e, 0x8e, 0xb2, 0x6d, 0x17, 0x8e, 0xcb, 0x03,
            0xa6, 0x1f, 0xbd, 0x37, 0x44, 0x0a, 0xce, 0xfc, 0x10, 0x64, 0x45, 0x63, 0x4e, 0x27, 0x64, 0x87,
            0x5f, 0x87, 0xa4, 0x14, 0x84, 0x11, 0xbd, 0xda, 0xe8, 0xac, 0xdf, 0x2c, 0x72, 0xd7, 0x58, 0xfc,
            0xc5, 0x15, 0x55, 0x70, 0x96, 0x3b, 0x90, 0xb0, 0x0c, 0x2b, 0x2c, 0x16, 0x2f, 0x45, 0x36, 0x22,
            0xd6, 0xf8, 0xc4, 0x0c, 0x3d, 0x84, 0xde, 0xe5, 0xcf, 0xc8, 0x67, 0x2c, 0xb3, 0xa0, 0x24, 0x87,
            0xd9, 0x93, 0x47, 0xa5, 0x06, 0x4b, 0xa1, 0xf0, 0xd8, 0x71, 0x65, 0x1f, 0x73, 0xd6, 0x0f, 0xde,
            0x0b, 0x46, 0x38, 0x91, 0x0f, 0x03, 0x67, 0xe8, 0x2e, 0x21, 0xa6, 0xb6, 0xa3, 0xd2, 0x96, 0x51,
            0x6c, 0x80, 0x44, 0xee, 0xe5, 0x85, 0x7c, 0xd0, 0x7d, 0xab, 0xc1, 0x63, 0x47, 0xcb, 0xf9, 0xa6,
            0x8c, 0xcb, 0x3e, 0x26, 0x55, 0x02, 0x03, 0x01, 0x20, 0x01, 0x02, 0x81, 0x80, 0x03, 0xc0, 0x5d,
            0x42, 0x21, 0xe5, 0xc8, 0x4b, 0x67, 0xcd, 0x08, 0x16, 0xcf, 0x35, 0xc5, 0xa6, 0xbc, 0x54, 0x09,
            0x26, 0x8f, 0xfa, 0xa9, 0x51, 0x4f, 0xa4, 0xe2, 0xfb, 0x34, 0x76, 0x57, 0xb9, 0x36, 0x67, 0x41,
            0xef, 0x0f, 0x3c, 0xee, 0x44, 0xc2, 0x2e, 0x31, 0xb0, 0x3d, 0xf3, 0xc2, 0xa0, 0x6e, 0x43, 0x74,
            0x79, 0xcd, 0xb0, 0xf2, 0x07, 0xaf, 0x20, 0xa7, 0x5c, 0xc9, 0x8b, 0xe0, 0x76, 0xa4, 0xfc, 0x18,
            0x29, 0x14, 0x6f, 0x2c, 0x32, 0x3b, 0x8f, 0x50, 0xaa, 0x8f, 0x9b, 0x50, 0xed, 0x3f, 0x63, 0x44,
            0xf4, 0xf2, 0x7c, 0x37, 0xdc, 0x8a, 0xc3, 0xd0, 0x31, 0xaa, 0x64, 0x7f, 0x88, 0xb0, 0x02, 0xfd,
            0xe1, 0x18, 0x0c, 0x6b, 0x11, 0xe7, 0x42, 0x81, 0xd9, 0x20, 0x6f, 0xc3, 0xbb, 0x5e, 0xa4, 0xfa,
            0x50, 0x37, 0xc9, 0xca, 0xb8, 0x41, 0x11, 0xec, 0x37, 0xbe, 0x51, 0xd9, 0x41, 0x02, 0x41, 0x20,
            0xe6, 0x39, 0x0c, 0x38, 0x5d, 0x02, 0x83, 0x01, 0x3c, 0x07, 0x90, 0xf9, 0x8e, 0x87, 0x78, 0x26,
            0x05, 0xef, 0x55, 0xe4, 0x8b, 0x4b, 0xa2, 0xaf, 0xd8, 0xcd, 0xe0, 0xa8, 0x20, 0xe6, 0x61, 0xd9,
            0xd3, 0x46, 0xe5, 0x54, 0x44, 0x4e, 0x36, 0x9a, 0x78, 0x43, 0xe1, 0xf5, 0xfa, 0x03, 0x0f, 0x73,
            0xe8, 0x02, 0xbf, 0xf5, 0x11, 0x6b, 0x63, 0x81, 0xe5, 0x1b, 0xd9, 0x09, 0x11, 0xbe, 0x91, 0x07,
            0x02, 0x41, 0x20, 0xc4, 0x65, 0xa3, 0xac, 0xa4, 0xb9, 0xe1, 0x58, 0xb0, 0x55, 0x91, 0x03, 0x47,
            0x59, 0x16, 0xb4, 0x13, 0x94, 0x61, 0x88, 0x99, 0x61, 0x25, 0xd2, 0x38, 0x0b, 0x02, 0x43, 0xe6,
            0x06, 0xcb, 0xad, 0xea, 0x4d, 0x89, 0x74, 0xba, 0x7e, 0x2d, 0xf4, 0x53, 0xa0, 0x35, 0x70, 0x38,
            0x23, 0xae, 0x50, 0xcf, 0x6f, 0x96, 0xef, 0x34, 0x72, 0xf6, 0x91, 0xb5, 0xea, 0x12, 0x5b, 0x88,
            0xa7, 0x62, 0xc3, 0x02, 0x40, 0x44, 0xeb, 0x41, 0x08, 0x42, 0x77, 0xb5, 0x06, 0x12, 0x95, 0x65,
            0x45, 0xff, 0xef, 0xf7, 0xfc, 0xd1, 0xfc, 0x02, 0xe7, 0x86, 0xf7, 0x9e, 0x7d, 0x2b, 0xd6, 0xfd,
            0xf0, 0xed, 0x4c, 0x0a, 0x70, 0xed, 0xf9, 0xa4, 0x3f, 0x64, 0xaf, 0x51, 0x53, 0x4e, 0xbb, 0x6b,
            0x92, 0x7a, 0x44, 0x9e, 0x44, 0xa1, 0x68, 0xe9, 0x54, 0x10, 0xe1, 0xe8, 0x09, 0x03, 0xb0, 0x21,
            0x79, 0xc0, 0x91, 0x1e, 0xd3, 0x02, 0x40, 0x27, 0xc0, 0x6e, 0xd5, 0x95, 0xed, 0xed, 0xb7, 0xb6,
            0xd1, 0x4a, 0xed, 0xbf, 0xf6, 0x3f, 0x4b, 0x29, 0x9f, 0x61, 0xf2, 0x75, 0x21, 0x1d, 0xb7, 0xa4,
            0x12, 0x4e, 0x5e, 0x11, 0xe4, 0x14, 0x3b, 0xda, 0x6d, 0xe1, 0x41, 0x98, 0x31, 0xcb, 0xf9, 0x65,
            0x27, 0xcb, 0x49, 0x30, 0x83, 0xc0, 0x57, 0xa8, 0xf4, 0xec, 0x36, 0xf4, 0x85, 0xc4, 0x74, 0x8f,
            0xbe, 0x96, 0x02, 0xa2, 0xf1, 0x46, 0x63, 0x02, 0x40, 0x29, 0xed, 0xe3, 0x32, 0xbc, 0xad, 0x5d,
            0xd4, 0x8e, 0x2b, 0x3a, 0x56, 0x39, 0x73, 0x62, 0x34, 0xa2, 0x89, 0xfc, 0xd2, 0x5c, 0x82, 0xb5,
            0x33, 0xd5, 0xcf, 0x6f, 0x98, 0xd0, 0xa5, 0xda, 0x0d, 0xc4, 0x1e, 0x6d, 0xc4, 0x2f, 0x4f, 0x69,
            0x02, 0x37, 0x33, 0xd1, 0xc4, 0xcd, 0x35, 0x49, 0x1c, 0xa1, 0x90, 0xfa, 0x70, 0xed, 0x59, 0xa7,
            0x58, 0x67, 0x71, 0xf4, 0x4a, 0x32, 0x96, 0x6c, 0x20
        }
    },
    {
        "1024 bit DER/PKCS8 encrypted", U_SECURITY_CREDENTIAL_CLIENT_KEY_PRIVATE,
        677, U_SECURITY_CREDENTIAL_TEST_PASSPHRASE,
        //lint -e785 Suppress Too few initializers
        {
            0x30, 0x82, 0x02, 0xa1, 0x30, 0x1b, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x05,
            0x03, 0x30, 0x0e, 0x04, 0x08, 0x99, 0xf7, 0x66, 0xde, 0x53, 0x78, 0x37, 0xc3, 0x02, 0x02, 0x08,
            0x20, 0x04, 0x82, 0x02, 0x80, 0xc4, 0x30, 0xfb, 0xfe, 0x75, 0xf8, 0x0a, 0x1f, 0x74, 0x10, 0x83,
            0xf8, 0xdb, 0xa6, 0xd4, 0x3e, 0x50, 0xaf, 0xcf, 0xcd, 0x70, 0x3f, 0xd3, 0xeb, 0xdf, 0xe9, 0x87,
            0xd6, 0x5a, 0xcd, 0x1a, 0x55, 0xea, 0x45, 0xa7, 0x5c, 0x7a, 0x66, 0x1e, 0x1b, 0x51, 0x79, 0xca,
            0x6f, 0x84, 0x21, 0xad, 0x60, 0x26, 0xda, 0x1c, 0x28, 0xb8, 0x44, 0x30, 0x4c, 0x99, 0x12, 0x7a,
            0x1d, 0xb3, 0xb9, 0x85, 0xa9, 0xfd, 0x19, 0x23, 0x0f, 0x3d, 0x5e, 0x4f, 0xe4, 0x32, 0x6a, 0x50,
            0xf4, 0xb6, 0xdd, 0x6f, 0xde, 0xf9, 0x62, 0x20, 0x8e, 0x41, 0x85, 0xe4, 0xea, 0xc1, 0x35, 0x6b,
            0xd6, 0xa7, 0xd1, 0x7c, 0x0b, 0xee, 0x1d, 0xd5, 0xe5, 0x03, 0x0f, 0x3f, 0x6c, 0x97, 0x02, 0xc8,
            0xfd, 0xef, 0x7f, 0xc3, 0x42, 0x57, 0x24, 0x31, 0x48, 0x5a, 0xf6, 0x6c, 0x3b, 0x0f, 0xbc, 0xc1,
            0xa2, 0x39, 0x11, 0x03, 0xef, 0xb8, 0x3d, 0xdc, 0xc9, 0x1a, 0xa1, 0x52, 0xa0, 0x1d, 0x8f, 0xbd,
            0x96, 0x43, 0x5c, 0x4d, 0x0a, 0x68, 0xc0, 0xba, 0xed, 0xb6, 0xd1, 0xcc, 0x20, 0x0c, 0x18, 0x01,
            0x6f, 0xcb, 0x8b, 0x3b, 0x04, 0x70, 0x01, 0x98, 0xf9, 0x32, 0x10, 0xb8, 0x8d, 0xa5, 0x6d, 0x91,
            0x52, 0x22, 0x15, 0x74, 0x12, 0x7f, 0x88, 0xb3, 0x6e, 0x20, 0xe5, 0xf7, 0x9d, 0x50, 0xb7, 0xea,
            0x7a, 0xfb, 0x29, 0xa7, 0x4f, 0xe2, 0xc5, 0x04, 0x97, 0xa1, 0x39, 0x16, 0xab, 0xfe, 0x3f, 0xfb,
            0x62, 0xfb, 0xfd, 0xb2, 0x39, 0x52, 0x67, 0x30, 0xad, 0x69, 0x78, 0x6a, 0xd0, 0x68, 0x9a, 0x94,
            0x5b, 0x56, 0xd3, 0x1e, 0x6d, 0x4c, 0xc6, 0x25, 0x1e, 0xc3, 0x7e, 0x38, 0xbe, 0xb9, 0xae, 0xc5,
            0x32, 0xe2, 0xed, 0x72, 0x2e, 0xe8, 0x5d, 0x0b, 0xee, 0x97, 0x40, 0x93, 0xc3, 0x0b, 0x67, 0x81,
            0xe6, 0x2d, 0xd9, 0x3d, 0x3a, 0x17, 0x6a, 0xf7, 0x5e, 0x87, 0xd5, 0x77, 0x9e, 0xf0, 0x16, 0x92,
            0xe7, 0x1b, 0xdb, 0xb9, 0x37, 0x0a, 0x32, 0x35, 0xc1, 0x5f, 0xa4, 0xaf, 0x2b, 0x69, 0x7a, 0xb7,
            0x2c, 0xf5, 0xf4, 0x7c, 0x03, 0xce, 0xa0, 0xaa, 0x61, 0xdb, 0x73, 0x2f, 0xaf, 0x33, 0x2b, 0x35,
            0x92, 0xf3, 0x45, 0xcd, 0xb5, 0x52, 0xd2, 0x41, 0x49, 0xff, 0xb3, 0x91, 0xcc, 0x0e, 0x6d, 0xbe,
            0x72, 0x98, 0x33, 0x0f, 0xca, 0xcc, 0x5e, 0x72, 0x51, 0xe8, 0x28, 0xae, 0xbb, 0xd0, 0xac, 0x33,
            0x7f, 0xb3, 0x81, 0xa7, 0xa1, 0x75, 0xa9, 0x54, 0x10, 0xca, 0x8a, 0x31, 0x21, 0x93, 0xd6, 0x37,
            0xc5, 0x67, 0x6a, 0xc2, 0xfb, 0x01, 0xf2, 0x1b, 0x20, 0x4a, 0xef, 0xd5, 0xe6, 0x9d, 0x8f, 0x13,
            0x89, 0x12, 0xbf, 0x10, 0x75, 0x86, 0x1a, 0x2f, 0x2a, 0x75, 0x8b, 0xcd, 0x47, 0x58, 0x4d, 0xee,
            0x8a, 0xaa, 0xe0, 0x6a, 0xbf, 0xd5, 0x4b, 0x13, 0xe3, 0x36, 0x27, 0x05, 0x3b, 0x8c, 0xb1, 0x5b,
            0x6b, 0x26, 0xfd, 0x99, 0x2e, 0x28, 0xc5, 0xd3, 0x8a, 0x25, 0xa6, 0xd9, 0x36, 0x60, 0xad, 0x17,
            0x62, 0x3a, 0x8f, 0x16, 0x59, 0x6e, 0x2a, 0x5b, 0x06, 0x3d, 0xd0, 0x8f, 0x04, 0xd9, 0x20, 0xf1,
            0x0c, 0x92, 0x7f, 0xe9, 0x8e, 0x98, 0xe9, 0xf3, 0x07, 0xb9, 0x67, 0x5d, 0x2d, 0x55, 0x0c, 0x21,
            0x9c, 0x7b, 0x2e, 0x8d, 0xc8, 0x62, 0x42, 0xb8, 0xd5, 0x68, 0xdc, 0x7c, 0x94, 0x58, 0x1e, 0xca,
            0x35, 0x5d, 0x88, 0xe3, 0xd2, 0xa7, 0xaa, 0x6f, 0xd2, 0x3e, 0xb0, 0x18, 0x75, 0x1c, 0xdc, 0xc6,
            0x1f, 0xf2, 0x56, 0x87, 0x8c, 0xdd, 0x8d, 0x0a, 0xbc, 0x51, 0xc0, 0x49, 0x06, 0x46, 0xe9, 0x0c,
            0xb9, 0x3b, 0xe6, 0x92, 0x12, 0x04, 0x1a, 0x8a, 0x8b, 0x90, 0x1a, 0xdd, 0xe4, 0x26, 0xad, 0x9b,
            0xca, 0x8c, 0x04, 0x81, 0x60, 0xa8, 0x48, 0x9e, 0x45, 0xe0, 0x44, 0x1c, 0x39, 0x2f, 0x4c, 0x05,
            0x47, 0x13, 0x7a, 0x28, 0x31, 0x6d, 0x59, 0x6f, 0x91, 0x6b, 0xd7, 0x43, 0x5a, 0x82, 0x6b, 0xb1,
            0x46, 0x62, 0xea, 0xa0, 0xd4, 0x09, 0x26, 0x6c, 0x97, 0x75, 0x19, 0xa1, 0x4d, 0x66, 0x58, 0x37,
            0x40, 0xd6, 0xa8, 0xf5, 0x2d, 0x51, 0xf5, 0x4c, 0x97, 0xe2, 0x11, 0xc1, 0x71, 0xec, 0xf0, 0xe0,
            0x57, 0x9d, 0x11, 0x7f, 0x01, 0x0a, 0xce, 0xaf, 0x85, 0x0a, 0xcb, 0x90, 0x21, 0x4a, 0x74, 0xe9,
            0x3a, 0x34, 0x54, 0x4e, 0x65, 0xca, 0xfa, 0x8a, 0xea, 0x13, 0x98, 0x3b, 0x82, 0xda, 0xa9, 0x12,
            0xfa, 0x7c, 0x7e, 0x25, 0x8c, 0xa1, 0x0b, 0x85, 0x06, 0xd0, 0xbc, 0x26, 0x85, 0x67, 0xc9, 0xc6,
            0x13, 0x78, 0xf2, 0xb2, 0xb2, 0xa9, 0xb5, 0x03, 0x04, 0x94, 0x52, 0xb7, 0x29, 0x4c, 0x61, 0xbb,
            0x89, 0xde, 0xfd, 0x4c, 0x51
        }
    },
    {
        "X.509 root/CA PEM", U_SECURITY_CREDENTIAL_ROOT_CA_X509,
        948, NULL,
        //lint -e785 Suppress Too few initializers
        {
            0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x42, 0x45, 0x47, 0x49, 0x4e, 0x20, 0x43, 0x45, 0x52, 0x54, 0x49,
            0x46, 0x49, 0x43, 0x41, 0x54, 0x45, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x0d, 0x0a, 0x4d, 0x49, 0x49,
            0x43, 0x68, 0x44, 0x43, 0x43, 0x41, 0x65, 0x32, 0x67, 0x41, 0x77, 0x49, 0x42, 0x41, 0x67, 0x49,
            0x4a, 0x41, 0x4d, 0x51, 0x76, 0x42, 0x61, 0x4b, 0x34, 0x68, 0x6b, 0x73, 0x76, 0x4d, 0x41, 0x30,
            0x47, 0x43, 0x53, 0x71, 0x47, 0x53, 0x49, 0x62, 0x33, 0x44, 0x51, 0x45, 0x42, 0x43, 0x77, 0x55,
            0x41, 0x4d, 0x46, 0x73, 0x78, 0x43, 0x7a, 0x41, 0x4a, 0x42, 0x67, 0x4e, 0x56, 0x0d, 0x0a, 0x42,
            0x41, 0x59, 0x54, 0x41, 0x6b, 0x64, 0x43, 0x4d, 0x52, 0x4d, 0x77, 0x45, 0x51, 0x59, 0x44, 0x56,
            0x51, 0x51, 0x49, 0x44, 0x41, 0x70, 0x54, 0x62, 0x32, 0x31, 0x6c, 0x4c, 0x56, 0x4e, 0x30, 0x59,
            0x58, 0x52, 0x6c, 0x4d, 0x52, 0x49, 0x77, 0x45, 0x41, 0x59, 0x44, 0x56, 0x51, 0x51, 0x48, 0x44,
            0x41, 0x6c, 0x44, 0x59, 0x57, 0x31, 0x69, 0x63, 0x6d, 0x6c, 0x6b, 0x5a, 0x32, 0x55, 0x78, 0x0d,
            0x0a, 0x44, 0x7a, 0x41, 0x4e, 0x42, 0x67, 0x4e, 0x56, 0x42, 0x41, 0x6f, 0x4d, 0x42, 0x6e, 0x55,
            0x74, 0x59, 0x6d, 0x78, 0x76, 0x65, 0x44, 0x45, 0x53, 0x4d, 0x42, 0x41, 0x47, 0x41, 0x31, 0x55,
            0x45, 0x41, 0x77, 0x77, 0x4a, 0x64, 0x57, 0x4a, 0x34, 0x62, 0x47, 0x6c, 0x69, 0x49, 0x47, 0x4e,
            0x68, 0x4d, 0x42, 0x34, 0x58, 0x44, 0x54, 0x49, 0x78, 0x4d, 0x44, 0x49, 0x78, 0x4d, 0x44, 0x45,
            0x32, 0x0d, 0x0a, 0x4e, 0x54, 0x49, 0x7a, 0x4e, 0x56, 0x6f, 0x58, 0x44, 0x54, 0x49, 0x32, 0x4d,
            0x44, 0x49, 0x77, 0x4f, 0x54, 0x45, 0x32, 0x4e, 0x54, 0x49, 0x7a, 0x4e, 0x56, 0x6f, 0x77, 0x57,
            0x7a, 0x45, 0x4c, 0x4d, 0x41, 0x6b, 0x47, 0x41, 0x31, 0x55, 0x45, 0x42, 0x68, 0x4d, 0x43, 0x52,
            0x30, 0x49, 0x78, 0x45, 0x7a, 0x41, 0x52, 0x42, 0x67, 0x4e, 0x56, 0x42, 0x41, 0x67, 0x4d, 0x43,
            0x6c, 0x4e, 0x76, 0x0d, 0x0a, 0x62, 0x57, 0x55, 0x74, 0x55, 0x33, 0x52, 0x68, 0x64, 0x47, 0x55,
            0x78, 0x45, 0x6a, 0x41, 0x51, 0x42, 0x67, 0x4e, 0x56, 0x42, 0x41, 0x63, 0x4d, 0x43, 0x55, 0x4e,
            0x68, 0x62, 0x57, 0x4a, 0x79, 0x61, 0x57, 0x52, 0x6e, 0x5a, 0x54, 0x45, 0x50, 0x4d, 0x41, 0x30,
            0x47, 0x41, 0x31, 0x55, 0x45, 0x43, 0x67, 0x77, 0x47, 0x64, 0x53, 0x31, 0x69, 0x62, 0x47, 0x39,
            0x34, 0x4d, 0x52, 0x49, 0x77, 0x0d, 0x0a, 0x45, 0x41, 0x59, 0x44, 0x56, 0x51, 0x51, 0x44, 0x44,
            0x41, 0x6c, 0x31, 0x59, 0x6e, 0x68, 0x73, 0x61, 0x57, 0x49, 0x67, 0x59, 0x32, 0x45, 0x77, 0x67,
            0x5a, 0x38, 0x77, 0x44, 0x51, 0x59, 0x4a, 0x4b, 0x6f, 0x5a, 0x49, 0x68, 0x76, 0x63, 0x4e, 0x41,
            0x51, 0x45, 0x42, 0x42, 0x51, 0x41, 0x44, 0x67, 0x59, 0x30, 0x41, 0x4d, 0x49, 0x47, 0x4a, 0x41,
            0x6f, 0x47, 0x42, 0x41, 0x4c, 0x43, 0x66, 0x0d, 0x0a, 0x46, 0x51, 0x36, 0x4f, 0x73, 0x6d, 0x30,
            0x58, 0x6a, 0x73, 0x73, 0x44, 0x70, 0x68, 0x2b, 0x39, 0x4e, 0x30, 0x51, 0x4b, 0x7a, 0x76, 0x77,
            0x51, 0x5a, 0x45, 0x56, 0x6a, 0x54, 0x69, 0x64, 0x6b, 0x68, 0x31, 0x2b, 0x48, 0x70, 0x42, 0x53,
            0x45, 0x45, 0x62, 0x33, 0x61, 0x36, 0x4b, 0x7a, 0x66, 0x4c, 0x48, 0x4c, 0x58, 0x57, 0x50, 0x7a,
            0x46, 0x46, 0x56, 0x56, 0x77, 0x6c, 0x6a, 0x75, 0x51, 0x0d, 0x0a, 0x73, 0x41, 0x77, 0x72, 0x4c,
            0x42, 0x59, 0x76, 0x52, 0x54, 0x59, 0x69, 0x31, 0x76, 0x6a, 0x45, 0x44, 0x44, 0x32, 0x45, 0x33,
            0x75, 0x58, 0x50, 0x79, 0x47, 0x63, 0x73, 0x73, 0x36, 0x41, 0x6b, 0x68, 0x39, 0x6d, 0x54, 0x52,
            0x36, 0x55, 0x47, 0x53, 0x36, 0x48, 0x77, 0x32, 0x48, 0x46, 0x6c, 0x48, 0x33, 0x50, 0x57, 0x44,
            0x39, 0x34, 0x4c, 0x52, 0x6a, 0x69, 0x52, 0x44, 0x77, 0x4e, 0x6e, 0x0d, 0x0a, 0x36, 0x43, 0x34,
            0x68, 0x70, 0x72, 0x61, 0x6a, 0x30, 0x70, 0x5a, 0x52, 0x62, 0x49, 0x42, 0x45, 0x37, 0x75, 0x57,
            0x46, 0x66, 0x4e, 0x42, 0x39, 0x71, 0x38, 0x46, 0x6a, 0x52, 0x38, 0x76, 0x35, 0x70, 0x6f, 0x7a,
            0x4c, 0x50, 0x69, 0x5a, 0x56, 0x41, 0x67, 0x4d, 0x42, 0x41, 0x41, 0x47, 0x6a, 0x55, 0x44, 0x42,
            0x4f, 0x4d, 0x42, 0x30, 0x47, 0x41, 0x31, 0x55, 0x64, 0x44, 0x67, 0x51, 0x57, 0x0d, 0x0a, 0x42,
            0x42, 0x53, 0x35, 0x59, 0x6c, 0x2b, 0x67, 0x51, 0x57, 0x36, 0x4d, 0x63, 0x6a, 0x67, 0x59, 0x75,
            0x47, 0x79, 0x41, 0x44, 0x37, 0x61, 0x62, 0x39, 0x63, 0x76, 0x4d, 0x6e, 0x54, 0x41, 0x66, 0x42,
            0x67, 0x4e, 0x56, 0x48, 0x53, 0x4d, 0x45, 0x47, 0x44, 0x41, 0x57, 0x67, 0x42, 0x53, 0x35, 0x59,
            0x6c, 0x2b, 0x67, 0x51, 0x57, 0x36, 0x4d, 0x63, 0x6a, 0x67, 0x59, 0x75, 0x47, 0x79, 0x41, 0x0d,
            0x0a, 0x44, 0x37, 0x61, 0x62, 0x39, 0x63, 0x76, 0x4d, 0x6e, 0x54, 0x41, 0x4d, 0x42, 0x67, 0x4e,
            0x56, 0x48, 0x52, 0x4d, 0x45, 0x42, 0x54, 0x41, 0x44, 0x41, 0x51, 0x48, 0x2f, 0x4d, 0x41, 0x30,
            0x47, 0x43, 0x53, 0x71, 0x47, 0x53, 0x49, 0x62, 0x33, 0x44, 0x51, 0x45, 0x42, 0x43, 0x77, 0x55,
            0x41, 0x41, 0x34, 0x47, 0x42, 0x41, 0x4b, 0x6a, 0x31, 0x36, 0x34, 0x69, 0x2b, 0x6c, 0x65, 0x4c,
            0x69, 0x0d, 0x0a, 0x34, 0x6e, 0x6a, 0x5a, 0x6e, 0x55, 0x35, 0x33, 0x4d, 0x37, 0x50, 0x69, 0x42,
            0x45, 0x6f, 0x63, 0x58, 0x59, 0x37, 0x2f, 0x49, 0x69, 0x6c, 0x70, 0x31, 0x51, 0x4f, 0x77, 0x32,
            0x47, 0x69, 0x66, 0x45, 0x31, 0x55, 0x37, 0x6a, 0x67, 0x64, 0x52, 0x65, 0x6a, 0x2b, 0x37, 0x44,
            0x4b, 0x4b, 0x79, 0x4c, 0x78, 0x63, 0x32, 0x4c, 0x35, 0x69, 0x45, 0x34, 0x48, 0x4c, 0x56, 0x4d,
            0x51, 0x70, 0x45, 0x0d, 0x0a, 0x4b, 0x79, 0x76, 0x6b, 0x45, 0x6d, 0x30, 0x7a, 0x43, 0x4f, 0x68,
            0x6b, 0x48, 0x50, 0x74, 0x6a, 0x64, 0x44, 0x73, 0x62, 0x38, 0x66, 0x58, 0x65, 0x59, 0x59, 0x44,
            0x6e, 0x75, 0x42, 0x6d, 0x75, 0x75, 0x75, 0x43, 0x51, 0x76, 0x79, 0x49, 0x46, 0x41, 0x66, 0x4a,
            0x4c, 0x4f, 0x61, 0x76, 0x42, 0x32, 0x66, 0x46, 0x62, 0x79, 0x42, 0x39, 0x67, 0x46, 0x78, 0x6b,
            0x63, 0x4a, 0x41, 0x4e, 0x6f, 0x0d, 0x0a, 0x2b, 0x68, 0x38, 0x7a, 0x48, 0x4a, 0x6c, 0x75, 0x44,
            0x56, 0x79, 0x4b, 0x64, 0x45, 0x2b, 0x59, 0x37, 0x58, 0x6b, 0x47, 0x36, 0x6d, 0x66, 0x6f, 0x36,
            0x65, 0x2f, 0x35, 0x47, 0x38, 0x75, 0x54, 0x0d, 0x0a, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x45, 0x4e,
            0x44, 0x20, 0x43, 0x45, 0x52, 0x54, 0x49, 0x46, 0x49, 0x43, 0x41, 0x54, 0x45, 0x2d, 0x2d, 0x2d,
            0x2d, 0x2d, 0x0d, 0x0a
        }
    },
    {
        "X.509 root/CA DER", U_SECURITY_CREDENTIAL_ROOT_CA_X509,
        648, NULL,
        //lint -e785 Suppress Too few initializers
        {
            0x30, 0x82, 0x02, 0x84, 0x30, 0x82, 0x01, 0xed, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x09, 0x20,
            0xc4, 0x2f, 0x05, 0xa2, 0xb8, 0x86, 0x4b, 0x2f, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86,
            0xf7, 0x0d, 0x01, 0x01, 0x0b, 0x05, 0x20, 0x30, 0x5b, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55,
            0x04, 0x06, 0x13, 0x02, 0x47, 0x42, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x08, 0x0c,
            0x0a, 0x53, 0x6f, 0x6d, 0x65, 0x2d, 0x53, 0x74, 0x61, 0x74, 0x65, 0x31, 0x12, 0x30, 0x10, 0x06,
            0x03, 0x55, 0x04, 0x07, 0x0c, 0x09, 0x43, 0x61, 0x6d, 0x62, 0x72, 0x69, 0x64, 0x67, 0x65, 0x31,
            0x0f, 0x30, 0x0d, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x0c, 0x06, 0x75, 0x2d, 0x62, 0x6c, 0x6f, 0x78,
            0x31, 0x12, 0x30, 0x10, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x09, 0x75, 0x62, 0x78, 0x6c, 0x69,
            0x62, 0x20, 0x63, 0x61, 0x30, 0x1e, 0x17, 0x0d, 0x32, 0x31, 0x30, 0x32, 0x31, 0x30, 0x31, 0x36,
            0x35, 0x32, 0x33, 0x35, 0x5a, 0x17, 0x0d, 0x32, 0x36, 0x30, 0x32, 0x30, 0x39, 0x31, 0x36, 0x35,
            0x32, 0x33, 0x35, 0x5a, 0x30, 0x5b, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13,
            0x02, 0x47, 0x42, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x08, 0x0c, 0x0a, 0x53, 0x6f,
            0x6d, 0x65, 0x2d, 0x53, 0x74, 0x61, 0x74, 0x65, 0x31, 0x12, 0x30, 0x10, 0x06, 0x03, 0x55, 0x04,
            0x07, 0x0c, 0x09, 0x43, 0x61, 0x6d, 0x62, 0x72, 0x69, 0x64, 0x67, 0x65, 0x31, 0x0f, 0x30, 0x0d,
            0x06, 0x03, 0x55, 0x04, 0x0a, 0x0c, 0x06, 0x75, 0x2d, 0x62, 0x6c, 0x6f, 0x78, 0x31, 0x12, 0x30,
            0x10, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x09, 0x75, 0x62, 0x78, 0x6c, 0x69, 0x62, 0x20, 0x63,
            0x61, 0x30, 0x81, 0x9f, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01,
            0x01, 0x05, 0x20, 0x03, 0x81, 0x8d, 0x20, 0x30, 0x81, 0x89, 0x02, 0x81, 0x81, 0x20, 0xb0, 0x9f,
            0x15, 0x0e, 0x8e, 0xb2, 0x6d, 0x17, 0x8e, 0xcb, 0x03, 0xa6, 0x1f, 0xbd, 0x37, 0x44, 0x0a, 0xce,
            0xfc, 0x10, 0x64, 0x45, 0x63, 0x4e, 0x27, 0x64, 0x87, 0x5f, 0x87, 0xa4, 0x14, 0x84, 0x11, 0xbd,
            0xda, 0xe8, 0xac, 0xdf, 0x2c, 0x72, 0xd7, 0x58, 0xfc, 0xc5, 0x15, 0x55, 0x70, 0x96, 0x3b, 0x90,
            0xb0, 0x0c, 0x2b, 0x2c, 0x16, 0x2f, 0x45, 0x36, 0x22, 0xd6, 0xf8, 0xc4, 0x0c, 0x3d, 0x84, 0xde,
            0xe5, 0xcf, 0xc8, 0x67, 0x2c, 0xb3, 0xa0, 0x24, 0x87, 0xd9, 0x93, 0x47, 0xa5, 0x06, 0x4b, 0xa1,
            0xf0, 0xd8, 0x71, 0x65, 0x1f, 0x73, 0xd6, 0x0f, 0xde, 0x0b, 0x46, 0x38, 0x91, 0x0f, 0x03, 0x67,
            0xe8, 0x2e, 0x21, 0xa6, 0xb6, 0xa3, 0xd2, 0x96, 0x51, 0x6c, 0x80, 0x44, 0xee, 0xe5, 0x85, 0x7c,
            0xd0, 0x7d, 0xab, 0xc1, 0x63, 0x47, 0xcb, 0xf9, 0xa6, 0x8c, 0xcb, 0x3e, 0x26, 0x55, 0x02, 0x03,
            0x01, 0x20, 0x01, 0xa3, 0x50, 0x30, 0x4e, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16,
            0x04, 0x14, 0xb9, 0x62, 0x5f, 0xa0, 0x41, 0x6e, 0x8c, 0x72, 0x38, 0x18, 0xb8, 0x6c, 0x80, 0x0f,
            0xb6, 0x9b, 0xf5, 0xcb, 0xcc, 0x9d, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04, 0x18, 0x30,
            0x16, 0x80, 0x14, 0xb9, 0x62, 0x5f, 0xa0, 0x41, 0x6e, 0x8c, 0x72, 0x38, 0x18, 0xb8, 0x6c, 0x80,
            0x0f, 0xb6, 0x9b, 0xf5, 0xcb, 0xcc, 0x9d, 0x30, 0x0c, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x04, 0x05,
            0x30, 0x03, 0x01, 0x01, 0xff, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01,
            0x01, 0x0b, 0x05, 0x20, 0x03, 0x81, 0x81, 0x20, 0xa8, 0xf5, 0xeb, 0x88, 0xbe, 0x95, 0xe2, 0xe2,
            0xe2, 0x78, 0xd9, 0x9d, 0x4e, 0x77, 0x33, 0xb3, 0xe2, 0x04, 0x4a, 0x1c, 0x5d, 0x8e, 0xff, 0x22,
            0x29, 0x69, 0xd5, 0x03, 0xb0, 0xd8, 0x68, 0x9f, 0x13, 0x55, 0x3b, 0x8e, 0x07, 0x51, 0x7a, 0x3f,
            0xbb, 0x0c, 0xa2, 0xb2, 0x2f, 0x17, 0x36, 0x2f, 0x98, 0x84, 0xe0, 0x72, 0xd5, 0x31, 0x0a, 0x44,
            0x2b, 0x2b, 0xe4, 0x12, 0x6d, 0x33, 0x08, 0xe8, 0x64, 0x1c, 0xfb, 0x63, 0x74, 0x3b, 0x1b, 0xf1,
            0xf5, 0xde, 0x61, 0x80, 0xe7, 0xb8, 0x19, 0xae, 0xba, 0xe0, 0x90, 0xbf, 0x22, 0x05, 0x01, 0xf2,
            0x4b, 0x39, 0xab, 0xc1, 0xd9, 0xf1, 0x5b, 0xc8, 0x1f, 0x60, 0x17, 0x19, 0x1c, 0x24, 0x03, 0x68,
            0xfa, 0x1f, 0x33, 0x1c, 0x99, 0x6e, 0x0d, 0x5c, 0x8a, 0x74, 0x4f, 0x98, 0xed, 0x79, 0x06, 0xea,
            0x67, 0xe8, 0xe9, 0xef, 0xf9, 0x1b, 0xcb, 0x93
        }
    },
    {
        "X.509 client PEM", U_SECURITY_CREDENTIAL_CLIENT_X509,
        824, NULL,
        //lint -e785 Suppress Too few initializers
        {
            0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x42, 0x45, 0x47, 0x49, 0x4e, 0x20, 0x43, 0x45, 0x52, 0x54, 0x49,
            0x46, 0x49, 0x43, 0x41, 0x54, 0x45, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x0d, 0x0a, 0x4d, 0x49, 0x49,
            0x43, 0x4b, 0x54, 0x43, 0x43, 0x41, 0x5a, 0x49, 0x43, 0x41, 0x51, 0x45, 0x77, 0x44, 0x51, 0x59,
            0x4a, 0x4b, 0x6f, 0x5a, 0x49, 0x68, 0x76, 0x63, 0x4e, 0x41, 0x51, 0x45, 0x4c, 0x42, 0x51, 0x41,
            0x77, 0x57, 0x7a, 0x45, 0x4c, 0x4d, 0x41, 0x6b, 0x47, 0x41, 0x31, 0x55, 0x45, 0x42, 0x68, 0x4d,
            0x43, 0x52, 0x30, 0x49, 0x78, 0x45, 0x7a, 0x41, 0x52, 0x42, 0x67, 0x4e, 0x56, 0x0d, 0x0a, 0x42,
            0x41, 0x67, 0x4d, 0x43, 0x6c, 0x4e, 0x76, 0x62, 0x57, 0x55, 0x74, 0x55, 0x33, 0x52, 0x68, 0x64,
            0x47, 0x55, 0x78, 0x45, 0x6a, 0x41, 0x51, 0x42, 0x67, 0x4e, 0x56, 0x42, 0x41, 0x63, 0x4d, 0x43,
            0x55, 0x4e, 0x68, 0x62, 0x57, 0x4a, 0x79, 0x61, 0x57, 0x52, 0x6e, 0x5a, 0x54, 0x45, 0x50, 0x4d,
            0x41, 0x30, 0x47, 0x41, 0x31, 0x55, 0x45, 0x43, 0x67, 0x77, 0x47, 0x64, 0x53, 0x31, 0x69, 0x0d,
            0x0a, 0x62, 0x47, 0x39, 0x34, 0x4d, 0x52, 0x49, 0x77, 0x45, 0x41, 0x59, 0x44, 0x56, 0x51, 0x51,
            0x44, 0x44, 0x41, 0x6c, 0x31, 0x59, 0x6e, 0x68, 0x73, 0x61, 0x57, 0x49, 0x67, 0x59, 0x32, 0x45,
            0x77, 0x48, 0x68, 0x63, 0x4e, 0x4d, 0x6a, 0x45, 0x77, 0x4d, 0x6a, 0x45, 0x77, 0x4d, 0x54, 0x63,
            0x78, 0x4f, 0x54, 0x41, 0x34, 0x57, 0x68, 0x63, 0x4e, 0x4d, 0x6a, 0x59, 0x77, 0x4d, 0x6a, 0x41,
            0x35, 0x0d, 0x0a, 0x4d, 0x54, 0x63, 0x78, 0x4f, 0x54, 0x41, 0x34, 0x57, 0x6a, 0x42, 0x66, 0x4d,
            0x51, 0x73, 0x77, 0x43, 0x51, 0x59, 0x44, 0x56, 0x51, 0x51, 0x47, 0x45, 0x77, 0x4a, 0x48, 0x51,
            0x6a, 0x45, 0x54, 0x4d, 0x42, 0x45, 0x47, 0x41, 0x31, 0x55, 0x45, 0x43, 0x41, 0x77, 0x4b, 0x55,
            0x32, 0x39, 0x74, 0x5a, 0x53, 0x31, 0x54, 0x64, 0x47, 0x46, 0x30, 0x5a, 0x54, 0x45, 0x53, 0x4d,
            0x42, 0x41, 0x47, 0x0d, 0x0a, 0x41, 0x31, 0x55, 0x45, 0x42, 0x77, 0x77, 0x4a, 0x51, 0x32, 0x46,
            0x74, 0x59, 0x6e, 0x4a, 0x70, 0x5a, 0x47, 0x64, 0x6c, 0x4d, 0x51, 0x38, 0x77, 0x44, 0x51, 0x59,
            0x44, 0x56, 0x51, 0x51, 0x4b, 0x44, 0x41, 0x5a, 0x31, 0x4c, 0x57, 0x4a, 0x73, 0x62, 0x33, 0x67,
            0x78, 0x46, 0x6a, 0x41, 0x55, 0x42, 0x67, 0x4e, 0x56, 0x42, 0x41, 0x4d, 0x4d, 0x44, 0x58, 0x56,
            0x69, 0x65, 0x47, 0x78, 0x70, 0x0d, 0x0a, 0x59, 0x69, 0x42, 0x6a, 0x62, 0x47, 0x6c, 0x6c, 0x62,
            0x6e, 0x51, 0x77, 0x67, 0x5a, 0x38, 0x77, 0x44, 0x51, 0x59, 0x4a, 0x4b, 0x6f, 0x5a, 0x49, 0x68,
            0x76, 0x63, 0x4e, 0x41, 0x51, 0x45, 0x42, 0x42, 0x51, 0x41, 0x44, 0x67, 0x59, 0x30, 0x41, 0x4d,
            0x49, 0x47, 0x4a, 0x41, 0x6f, 0x47, 0x42, 0x41, 0x4c, 0x43, 0x66, 0x46, 0x51, 0x36, 0x4f, 0x73,
            0x6d, 0x30, 0x58, 0x6a, 0x73, 0x73, 0x44, 0x0d, 0x0a, 0x70, 0x68, 0x2b, 0x39, 0x4e, 0x30, 0x51,
            0x4b, 0x7a, 0x76, 0x77, 0x51, 0x5a, 0x45, 0x56, 0x6a, 0x54, 0x69, 0x64, 0x6b, 0x68, 0x31, 0x2b,
            0x48, 0x70, 0x42, 0x53, 0x45, 0x45, 0x62, 0x33, 0x61, 0x36, 0x4b, 0x7a, 0x66, 0x4c, 0x48, 0x4c,
            0x58, 0x57, 0x50, 0x7a, 0x46, 0x46, 0x56, 0x56, 0x77, 0x6c, 0x6a, 0x75, 0x51, 0x73, 0x41, 0x77,
            0x72, 0x4c, 0x42, 0x59, 0x76, 0x52, 0x54, 0x59, 0x69, 0x0d, 0x0a, 0x31, 0x76, 0x6a, 0x45, 0x44,
            0x44, 0x32, 0x45, 0x33, 0x75, 0x58, 0x50, 0x79, 0x47, 0x63, 0x73, 0x73, 0x36, 0x41, 0x6b, 0x68,
            0x39, 0x6d, 0x54, 0x52, 0x36, 0x55, 0x47, 0x53, 0x36, 0x48, 0x77, 0x32, 0x48, 0x46, 0x6c, 0x48,
            0x33, 0x50, 0x57, 0x44, 0x39, 0x34, 0x4c, 0x52, 0x6a, 0x69, 0x52, 0x44, 0x77, 0x4e, 0x6e, 0x36,
            0x43, 0x34, 0x68, 0x70, 0x72, 0x61, 0x6a, 0x30, 0x70, 0x5a, 0x52, 0x0d, 0x0a, 0x62, 0x49, 0x42,
            0x45, 0x37, 0x75, 0x57, 0x46, 0x66, 0x4e, 0x42, 0x39, 0x71, 0x38, 0x46, 0x6a, 0x52, 0x38, 0x76,
            0x35, 0x70, 0x6f, 0x7a, 0x4c, 0x50, 0x69, 0x5a, 0x56, 0x41, 0x67, 0x4d, 0x42, 0x41, 0x41, 0x45,
            0x77, 0x44, 0x51, 0x59, 0x4a, 0x4b, 0x6f, 0x5a, 0x49, 0x68, 0x76, 0x63, 0x4e, 0x41, 0x51, 0x45,
            0x4c, 0x42, 0x51, 0x41, 0x44, 0x67, 0x59, 0x45, 0x41, 0x48, 0x77, 0x31, 0x56, 0x0d, 0x0a, 0x63,
            0x49, 0x51, 0x64, 0x4a, 0x52, 0x36, 0x45, 0x45, 0x54, 0x32, 0x5a, 0x77, 0x32, 0x37, 0x55, 0x79,
            0x53, 0x6c, 0x41, 0x32, 0x77, 0x57, 0x45, 0x34, 0x47, 0x38, 0x56, 0x41, 0x4c, 0x31, 0x46, 0x69,
            0x48, 0x67, 0x6a, 0x41, 0x59, 0x42, 0x42, 0x78, 0x61, 0x31, 0x7a, 0x61, 0x6e, 0x41, 0x61, 0x57,
            0x65, 0x6a, 0x36, 0x68, 0x56, 0x58, 0x6e, 0x31, 0x71, 0x68, 0x53, 0x32, 0x34, 0x4b, 0x4e, 0x0d,
            0x0a, 0x45, 0x76, 0x6f, 0x34, 0x51, 0x34, 0x42, 0x71, 0x71, 0x70, 0x65, 0x73, 0x33, 0x6c, 0x72,
            0x36, 0x75, 0x2f, 0x54, 0x58, 0x51, 0x52, 0x67, 0x54, 0x68, 0x77, 0x53, 0x6e, 0x4a, 0x56, 0x4d,
            0x4c, 0x74, 0x61, 0x6a, 0x67, 0x53, 0x4a, 0x66, 0x73, 0x71, 0x34, 0x6c, 0x56, 0x4d, 0x4f, 0x59,
            0x75, 0x5a, 0x52, 0x50, 0x71, 0x69, 0x48, 0x47, 0x64, 0x4a, 0x63, 0x69, 0x6c, 0x61, 0x4b, 0x37,
            0x54, 0x0d, 0x0a, 0x6a, 0x34, 0x55, 0x4e, 0x78, 0x67, 0x68, 0x79, 0x79, 0x49, 0x44, 0x58, 0x46,
            0x69, 0x79, 0x56, 0x30, 0x45, 0x5a, 0x42, 0x57, 0x31, 0x72, 0x75, 0x6c, 0x61, 0x4c, 0x6c, 0x34,
            0x64, 0x73, 0x2b, 0x70, 0x61, 0x47, 0x4e, 0x38, 0x69, 0x45, 0x3d, 0x0d, 0x0a, 0x2d, 0x2d, 0x2d,
            0x2d, 0x2d, 0x45, 0x4e, 0x44, 0x20, 0x43, 0x45, 0x52, 0x54, 0x49, 0x46, 0x49, 0x43, 0x41, 0x54,
            0x45, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x0d, 0x0a
        }
    },
    {
        "X.509 client DER", U_SECURITY_CREDENTIAL_CLIENT_X509,
        557, NULL,
        //lint -e785 Suppress Too few initializers
        {
            0x30, 0x82, 0x02, 0x29, 0x30, 0x82, 0x01, 0x92, 0x02, 0x01, 0x01, 0x30, 0x0d, 0x06, 0x09, 0x2a,
            0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0b, 0x05, 0x20, 0x30, 0x5b, 0x31, 0x0b, 0x30, 0x09,
            0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x47, 0x42, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55,
            0x04, 0x08, 0x0c, 0x0a, 0x53, 0x6f, 0x6d, 0x65, 0x2d, 0x53, 0x74, 0x61, 0x74, 0x65, 0x31, 0x12,
            0x30, 0x10, 0x06, 0x03, 0x55, 0x04, 0x07, 0x0c, 0x09, 0x43, 0x61, 0x6d, 0x62, 0x72, 0x69, 0x64,
            0x67, 0x65, 0x31, 0x0f, 0x30, 0x0d, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x0c, 0x06, 0x75, 0x2d, 0x62,
            0x6c, 0x6f, 0x78, 0x31, 0x12, 0x30, 0x10, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x09, 0x75, 0x62,
            0x78, 0x6c, 0x69, 0x62, 0x20, 0x63, 0x61, 0x30, 0x1e, 0x17, 0x0d, 0x32, 0x31, 0x30, 0x32, 0x31,
            0x30, 0x31, 0x37, 0x31, 0x39, 0x30, 0x38, 0x5a, 0x17, 0x0d, 0x32, 0x36, 0x30, 0x32, 0x30, 0x39,
            0x31, 0x37, 0x31, 0x39, 0x30, 0x38, 0x5a, 0x30, 0x5f, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55,
            0x04, 0x06, 0x13, 0x02, 0x47, 0x42, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x08, 0x0c,
            0x0a, 0x53, 0x6f, 0x6d, 0x65, 0x2d, 0x53, 0x74, 0x61, 0x74, 0x65, 0x31, 0x12, 0x30, 0x10, 0x06,
            0x03, 0x55, 0x04, 0x07, 0x0c, 0x09, 0x43, 0x61, 0x6d, 0x62, 0x72, 0x69, 0x64, 0x67, 0x65, 0x31,
            0x0f, 0x30, 0x0d, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x0c, 0x06, 0x75, 0x2d, 0x62, 0x6c, 0x6f, 0x78,
            0x31, 0x16, 0x30, 0x14, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x0d, 0x75, 0x62, 0x78, 0x6c, 0x69,
            0x62, 0x20, 0x63, 0x6c, 0x69, 0x65, 0x6e, 0x74, 0x30, 0x81, 0x9f, 0x30, 0x0d, 0x06, 0x09, 0x2a,
            0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x20, 0x03, 0x81, 0x8d, 0x20, 0x30, 0x81,
            0x89, 0x02, 0x81, 0x81, 0x20, 0xb0, 0x9f, 0x15, 0x0e, 0x8e, 0xb2, 0x6d, 0x17, 0x8e, 0xcb, 0x03,
            0xa6, 0x1f, 0xbd, 0x37, 0x44, 0x0a, 0xce, 0xfc, 0x10, 0x64, 0x45, 0x63, 0x4e, 0x27, 0x64, 0x87,
            0x5f, 0x87, 0xa4, 0x14, 0x84, 0x11, 0xbd, 0xda, 0xe8, 0xac, 0xdf, 0x2c, 0x72, 0xd7, 0x58, 0xfc,
            0xc5, 0x15, 0x55, 0x70, 0x96, 0x3b, 0x90, 0xb0, 0x0c, 0x2b, 0x2c, 0x16, 0x2f, 0x45, 0x36, 0x22,
            0xd6, 0xf8, 0xc4, 0x0c, 0x3d, 0x84, 0xde, 0xe5, 0xcf, 0xc8, 0x67, 0x2c, 0xb3, 0xa0, 0x24, 0x87,
            0xd9, 0x93, 0x47, 0xa5, 0x06, 0x4b, 0xa1, 0xf0, 0xd8, 0x71, 0x65, 0x1f, 0x73, 0xd6, 0x0f, 0xde,
            0x0b, 0x46, 0x38, 0x91, 0x0f, 0x03, 0x67, 0xe8, 0x2e, 0x21, 0xa6, 0xb6, 0xa3, 0xd2, 0x96, 0x51,
            0x6c, 0x80, 0x44, 0xee, 0xe5, 0x85, 0x7c, 0xd0, 0x7d, 0xab, 0xc1, 0x63, 0x47, 0xcb, 0xf9, 0xa6,
            0x8c, 0xcb, 0x3e, 0x26, 0x55, 0x02, 0x03, 0x01, 0x20, 0x01, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86,
            0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0b, 0x05, 0x20, 0x03, 0x81, 0x81, 0x20, 0x1f, 0x0d, 0x55,
            0x70, 0x84, 0x1d, 0x25, 0x1e, 0x84, 0x11, 0x3d, 0x99, 0xc3, 0x6e, 0xd4, 0xc9, 0x29, 0x40, 0xdb,
            0x05, 0x84, 0xe0, 0x6f, 0x15, 0x20, 0xbd, 0x45, 0x88, 0x78, 0x23, 0x01, 0x80, 0x41, 0xc5, 0xad,
            0x73, 0x6a, 0x70, 0x1a, 0x59, 0xe8, 0xfa, 0x85, 0x55, 0xe7, 0xd6, 0xa8, 0x52, 0xdb, 0x82, 0x8d,
            0x12, 0xfa, 0x38, 0x43, 0x80, 0x6a, 0xaa, 0x97, 0xac, 0xde, 0x5a, 0xfa, 0xbb, 0xf4, 0xd7, 0x41,
            0x18, 0x13, 0x87, 0x04, 0xa7, 0x25, 0x53, 0x0b, 0xb5, 0xa8, 0xe0, 0x48, 0x97, 0xec, 0xab, 0x89,
            0x55, 0x30, 0xe6, 0x2e, 0x65, 0x13, 0xea, 0x88, 0x71, 0x9d, 0x25, 0xc8, 0xa5, 0x68, 0xae, 0xd3,
            0x8f, 0x85, 0x0d, 0xc6, 0x08, 0x72, 0xc8, 0x80, 0xd7, 0x16, 0x2c, 0x95, 0xd0, 0x46, 0x41, 0x5b,
            0x5a, 0xee, 0x95, 0xa2, 0xe5, 0xe1, 0xdb, 0x3e, 0xa5, 0xa1, 0x8d, 0xf2, 0x21
        }
    }
};

// Number of items in the gUSecurityCredentialTestFormat array.
const size_t gUSecurityCredentialTestFormatSize =
    sizeof(gUSecurityCredentialTestFormat) / sizeof(gUSecurityCredentialTestFormat[0]);

#endif

// End of file
