/*
 * Copyright 2019-2024 u-blox
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/** @file
 * @brief The application entry point for the Linux platform.  Starts
 * the platform and calls Unity to run the selected examples/tests.
 */

#ifdef U_CFG_OVERRIDE
# include "u_cfg_override.h" // For a customer's configuration override
#endif

#include "stddef.h"    // NULL, size_t etc.
#include "stdint.h"    // int32_t etc.
#include "stdbool.h"
#include "string.h"
#include "stdlib.h"    // getenv

#include "u_cfg_sw.h"
#include "u_cfg_os_platform_specific.h"
#include "u_cfg_app_platform_specific.h"
#include "u_cfg_test_platform_specific.h"

#include "u_error_common.h"

#include "u_port.h"
#include "u_port_os.h"
#include "u_port_debug.h"

#include "u_debug_utils.h"

/* ----------------------------------------------------------------
 * COMPILE-TIME MACROS
 * -------------------------------------------------------------- */

/* ----------------------------------------------------------------
 * TYPES
 * -------------------------------------------------------------- */

/* ----------------------------------------------------------------
 * VARIABLES
 * -------------------------------------------------------------- */

// This is intentionally a bit hidden and comes from u_port_debug.c
extern int32_t gStdoutCounter;

/* ----------------------------------------------------------------
 * STATIC FUNCTIONS
 * -------------------------------------------------------------- */

// The task within which the examples and tests run.
static void appTask(void *pParam)
{
    (void) pParam;

    uPortInit();

#if U_CFG_TEST_ENABLE_INACTIVITY_DETECTOR
    // Normally this would be called before uPortInit() but
    // in the Linux case there is a list of threads set up
    // by uPortInit() that have to be sorted before mutex
    // debug can work
    uDebugUtilsInitInactivityDetector(&gStdoutCounter);
#endif

#ifdef U_CFG_MUTEX_DEBUG
    // Has to be after uPortInit() for the same reason as above
    uMutexDebugInit();
    uMutexDebugWatchdog(uMutexDebugPrint, NULL,
                        U_MUTEX_DEBUG_WATCHDOG_TIMEOUT_SECONDS);
#endif

    uPortLog("\n\nU_APP: application task started.\n");

    UNITY_BEGIN();

    uPortLog("U_APP: functions available:\n\n");
    uRunnerPrintAll("U_APP: ");
    const char *pEnvVar = getenv("U_CFG_APP_FILTER");
    if ((pEnvVar != NULL) && (strlen(pEnvVar) > 0)) {
        uPortLog("U_APP: * runtime function filter specified.\n");
        uPortLog("U_APP: running functions that begin with \"%s\".\n", pEnvVar);
        uRunnerRunFiltered(pEnvVar, "U_APP: ");
    } else {
#ifdef U_CFG_APP_FILTER
        uPortLog("U_APP: running functions that begin with \"%s\".\n",
                 U_PORT_STRINGIFY_QUOTED(U_CFG_APP_FILTER));
        uRunnerRunFiltered(U_PORT_STRINGIFY_QUOTED(U_CFG_APP_FILTER), "U_APP: ");
#else
        uPortLog("U_APP: running all functions.\n");
        uRunnerRunAll("U_APP: ");
#endif
    }

    // The things that we have run may have
    // called deinit so call init again here.
    uPortInit();

    // Don't show the unity completion if this is a spawned test.
    pEnvVar = getenv("U_CFG_TEST_SPAWNED");
    if ((pEnvVar == NULL) || (strlen(pEnvVar) == 0)) {
        UNITY_END();
    }

    uPortLog("\n\nU_APP: application task ended.\n");
    uPortDeinit();
}

/* ----------------------------------------------------------------
 * PUBLIC FUNCTIONS
 * -------------------------------------------------------------- */

// Unity setUp() function.
void setUp(void)
{
    // Nothing to do
}

// Unity tearDown() function.
void tearDown(void)
{
    // Nothing to do
}

void testFail(void)
{
    // Nothing to do
}

// Entry point
int main(void)
{
    // Start the platform to run the tests
    return uPortPlatformStart(appTask, NULL,
                              U_CFG_OS_APP_TASK_STACK_SIZE_BYTES,
                              U_CFG_OS_APP_TASK_PRIORITY);
}

// End of file
